/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-kopfzahl.
 *
 * Ipspiel21-kopfzahl is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-kopfzahl is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-kopfzahl. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahl.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.evolution.EvolutionPlayerImpl;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlPlayer;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlState;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.SaidMove;

/**
 * Implements {@link KopfzahlPlayer}.
 */
final class KopfzahlPlayerImpl extends EvolutionPlayerImpl implements KopfzahlPlayer {

    /**
     * The associated game state.
     */
    private final KopfzahlState gameState;
    /**
     * The name of this player.
     */
    private final String name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<SaidMove, Map<SaidMove, Double>> possibleOutcomes;

    /**
     * Creates a Kopfzahl player.
     *
     * @param gameState        The associated game state.
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    KopfzahlPlayerImpl(final KopfzahlState gameState, final String name,
            final Map<SaidMove, Map<SaidMove, Double>> possibleOutcomes) {
        this.gameState = gameState;
        this.name = Objects.requireNonNull(name, "name");
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
    }

    @Override
    public String toString() {
        return String.format("KopfzahlPlayer[name=%s, state=%s, outcome=%s]", this.name, this.getState(),
                this.getOutcome());
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof KopfzahlPlayerImpl) {
            final KopfzahlPlayerImpl other = (KopfzahlPlayerImpl) obj;
            return this.name.equals(other.name) && this.getState().equals(other.getState())
                    && this.getOutcome().equals(other.getOutcome());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return this.name.hashCode();
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public Map<SaidMove, Map<SaidMove, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public KopfzahlPlayer deepCopy(final KopfzahlState newGameState) {
        return new KopfzahlPlayerImpl(newGameState, this.name, this.possibleOutcomes);
    }

    @Override
    public PlayerState getState() {
        return this.gameState.getPlayerState(this.name);
    }

    @Override
    public Optional<Double> getOutcome() {
        return this.gameState.getPlayerOutcome(this.name);
    }
}
