/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-tictactoe-strategy-random.
 *
 * ipspiel21-tictactoe-strategy-random is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel21-tictactoe-strategy-random is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel21-tictactoe-strategy-random. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.tictactoe.strategy.random;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeField;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeFieldState;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToePlayer;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeState;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeStrategy;
import de.fhdw.gaming.ipspiel21.tictactoe.core.moves.TicTacToeMove;
import de.fhdw.gaming.ipspiel21.tictactoe.core.moves.factory.TicTacToeMoveFactory;

/**
 * Implements {@link TicTacToeStrategy} by randomly choosing the next possible move.
 */
public final class TicTacToeRandomMoveStrategy implements TicTacToeStrategy {

    /**
     * The factory for creating TicTacToe moves.
     */
    private final TicTacToeMoveFactory moveFactory;

    /**
     * Creates an {@link TicTacToeRandomMoveStrategy}.
     *
     * @param moveFactory The factory for creating TicTacToe moves.
     */
    TicTacToeRandomMoveStrategy(final TicTacToeMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    /**
     * Chooses some random move.
     */
    @Override
    public Optional<TicTacToeMove> computeNextMove(final int gameId, final TicTacToePlayer player,
            final TicTacToeState state) throws GameException {

        final List<TicTacToeField> fields = new ArrayList<>(
                state.getBoard().getFieldsBeing(TicTacToeFieldState.EMPTY).values());

        if (fields.isEmpty()) {
            return Optional.empty();
        } else {
            final int index = new Random().nextInt(fields.size());
            final TicTacToeField field = fields.get(index);
            return Optional.of(this.moveFactory.createPlaceMarkMove(player.isUsingCrosses(), field.getPosition()));
        }
    }

    @Override
    public String toString() {
        return TicTacToeRandomMoveStrategy.class.getSimpleName();
    }
}
