/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-Dilemma.
 *
 * Ipspiel21-Dil is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-Dil is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-Dilemma. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayer;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayerBuilder;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaState;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.AbstractDilemmaMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaBeSilentMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaConfessMove;

/**
 * the implementation for a Dilemmaplayer.
 */
final class DilemmaPlayerBuilderImpl implements DilemmaPlayerBuilder {

    /**
     * Default outcome value.
     */
    public static final Double DEFAULT_OUTCOME_SILENT_CONFESS = -10.0;
    /**
     * Default outcome value.
     */
    public static final Double DEFAULT_OUTCOME_CONFESS_SILENT = 0.0;
    /**
     * Default outcome value.
     */
    public static final Double DEFAULT_OUTCOME_SILENT_SILENT = -1.0;
    /**
     * Default outcome value.
     */
    public static final Double DEFAULT_OUTCOME_CONFESS_CONFESS = -8.0;
    /**
     * Default Name Counter, will be used for applying default values.
     */
    private static int defaultNameCounter = 1;

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player @key1 =
     * {@link AbstractDilemmaMove}, the key for the second-level map is the answer of the second player @key2 =
     * {@link AbstractDilemmaMove}.
     */
    private Optional<Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>>> possibleOutcomes;
    /** The associated Dilemma move. */
    private AbstractDilemmaMove move;

    /**
     * Creates an {@link DilemmaPlayerBuilderImpl}.
     */
    DilemmaPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public DilemmaPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public DilemmaPlayerBuilder changePossibleOutcomes(
            final Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public DilemmaPlayer build(final DilemmaState state) throws GameException {
        return new DilemmaPlayerImpl(state, this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>>
            checkPossibleOutcomes(final Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>> outcomes) {
        return outcomes;
    }

    /**
     * Getter for the associated move.
     * 
     * @return the current move.
     */
    public AbstractDilemmaMove getMove() {
        return move;
    }

    /**
     * Setter for the associated move.
     * 
     * @param move the move which has to be set.
     */
    public void setMove(final AbstractDilemmaMove move) {
        this.move = move;
    }

    /**
     * Creates a default dilemma player builder.
     * 
     */
    @Override
    public DilemmaPlayerBuilder defaultPlayerBuilder() {
        this.name = Optional.of(String.valueOf(DilemmaPlayerBuilderImpl.defaultNameCounter++));
        this.possibleOutcomes = Optional.of(new HashMap<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>>(2) {

            /**
             * 
             */
            private static final long serialVersionUID = 1L;

            {
                put(new DilemmaBeSilentMove(), new HashMap<AbstractDilemmaMove, Double>(3) {
                    /**
                     * 
                     */
                    private static final long serialVersionUID = 1L;

                    {
                        put(new DilemmaBeSilentMove(), DilemmaPlayerBuilderImpl.DEFAULT_OUTCOME_SILENT_SILENT);
                        put(new DilemmaConfessMove(), DilemmaPlayerBuilderImpl.DEFAULT_OUTCOME_SILENT_CONFESS);
                    }
                });
                put(new DilemmaConfessMove(), new HashMap<AbstractDilemmaMove, Double>(3) {
                    /**
                     * 
                     */
                    private static final long serialVersionUID = 1L;

                    {
                        put(new DilemmaConfessMove(), DilemmaPlayerBuilderImpl.DEFAULT_OUTCOME_CONFESS_CONFESS);
                        put(new DilemmaBeSilentMove(), DilemmaPlayerBuilderImpl.DEFAULT_OUTCOME_CONFESS_SILENT);
                    }
                });
            }
        });
        return this;
    }
}
