/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-kopfzahl.
 *
 * Ipspiel21-kopfzahl is free software: you can redistribute it 
 * and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-kopfzahl is distributed in the hope that it will 
 * be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-kopfzahl. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahl.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlGameBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlGameBuilderFactory;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlPlayerBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlStrategy;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.SaidMove;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.factory.KopfzahlDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.factory.KopfzahlStrategyFactory;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.factory.KopfzahlStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.kopfzahl.moves.factory.KopfzahlMoveFactory;
import de.fhdw.gaming.ipspiel21.kopfzahl.moves.impl.KopfzahlDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a Kopfzahl game builder.
 */
public final class KopfzahlGameBuilderFactoryImpl implements KopfzahlGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Kopfzahl strategies.
     */
    private final Set<KopfzahlStrategy> strategies;

    /**
     * Creates a Kopfzahl game factory. Kopfzahl strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public KopfzahlGameBuilderFactoryImpl() {
        this(new KopfzahlDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Kopfzahl game factory.
     *
     * @param strategyFactoryProvider The {@link KopfzahlStrategyFactoryProvider} for loading Kopfzahl strategies.
     */
    KopfzahlGameBuilderFactoryImpl(final KopfzahlStrategyFactoryProvider strategyFactoryProvider) {
        final KopfzahlMoveFactory moveFactory = new KopfzahlDefaultMoveFactory();

        final List<KopfzahlStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final KopfzahlStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Kopfzahl";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return KopfzahlGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return KopfzahlGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public KopfzahlGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final KopfzahlGameBuilder gameBuilder = new KopfzahlGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String, Object> gameData = inputProvider
                    .needInteger(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(KopfzahlGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(KopfzahlGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                    .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayer1Data(firstPlayerInputProvider, "Player 1");
            final KopfzahlPlayerBuilder firstPlayerBuilder = this.initPlayerBuilder(gameBuilder.createPlayerBuilder(),
                    firstPlayerData);
            final KopfzahlStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayerBuilder(firstPlayerBuilder, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayer2Data(secondPlayerInputProvider, "Player 2");
            final KopfzahlPlayerBuilder secondPlayerBuilder = this.initPlayerBuilder(gameBuilder.createPlayerBuilder(),
                    secondPlayerData);
            final KopfzahlStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayerBuilder(secondPlayerBuilder, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Kopfzahl game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayer1Data(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(GameBuilderFactory.PARAM_PLAYER_NAME, "Name", Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KOPF,
                        "Player's outcome on Kopf/Kopf", Optional.of(1))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_ZAHL,
                        "Player's outcome on Kopf/Zahl", Optional.of(-1))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KOPF,
                        "Player's outcome on Zahl/Kopf", Optional.of(-1))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_ZAHL,
                        "Player's outcome on Zahl/Zahl", Optional.of(1))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayer2Data(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(GameBuilderFactory.PARAM_PLAYER_NAME, "Name", Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KOPF,
                        "Player's outcome on Kopf/Kopf", Optional.of(-1))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_ZAHL,
                        "Player's outcome on Kopf/Zahl", Optional.of(1))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KOPF,
                        "Player's outcome on Zahl/Kopf", Optional.of(1))
                .needInteger(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_ZAHL,
                        "Player's outcome on Zahl/Zahl", Optional.of(-1))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Initialises a Kopfzahl player builder.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return {@code playerBuilder}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private KopfzahlPlayerBuilder initPlayerBuilder(final KopfzahlPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<SaidMove, Map<SaidMove, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<SaidMove, Double> possibleOutcomesKopf = new LinkedHashMap<>();
        possibleOutcomesKopf.put(SaidMove.KOPF,
                (double) (Integer) playerData.get(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KOPF));
        possibleOutcomesKopf.put(SaidMove.ZAHL,
                (double) (Integer) playerData.get(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_ZAHL));
        possibleOutcomes.put(SaidMove.KOPF, possibleOutcomesKopf);

        final Map<SaidMove, Double> possibleOutcomesZahl = new LinkedHashMap<>();
        possibleOutcomesZahl.put(SaidMove.KOPF,
                (double) (Integer) playerData.get(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KOPF));
        possibleOutcomesZahl.put(SaidMove.ZAHL,
                (double) (Integer) playerData.get(KopfzahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_ZAHL));
        possibleOutcomes.put(SaidMove.ZAHL, possibleOutcomesZahl);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes);
    }

    /**
     * Returns a Kopfzahl strategy.
     *
     * @param playerData The requested player data.
     * @return The Kopfzahl strategy.
     */
    private KopfzahlStrategy getStrategy(final Map<String, Object> playerData) {
        return (KopfzahlStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
