/*
 * Copyright © 2021-2022 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel22-demo.
 *
 * Ipspiel22-demo is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel22-demo is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel22-demo. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel22.demo.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel22.demo.domain.DemoGameBuilder;
import de.fhdw.gaming.ipspiel22.demo.domain.DemoGameBuilderFactory;
import de.fhdw.gaming.ipspiel22.demo.domain.DemoPlayer;
import de.fhdw.gaming.ipspiel22.demo.domain.DemoPlayerBuilder;
import de.fhdw.gaming.ipspiel22.demo.domain.DemoStrategy;
import de.fhdw.gaming.ipspiel22.demo.domain.factory.DemoDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.demo.domain.factory.DemoStrategyFactory;
import de.fhdw.gaming.ipspiel22.demo.domain.factory.DemoStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.demo.moves.factory.DemoMoveFactory;
import de.fhdw.gaming.ipspiel22.demo.moves.impl.DemoDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a Demo game builder.
 */
public final class DemoGameBuilderFactoryImpl implements DemoGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Demo strategies.
     */
    private final Set<DemoStrategy> strategies;

    /**
     * Creates a Demo game factory. Demo strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public DemoGameBuilderFactoryImpl() {
        this(new DemoDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Demo game factory.
     *
     * @param strategyFactoryProvider The {@link DemoStrategyFactoryProvider} for loading Demo strategies.
     */
    DemoGameBuilderFactoryImpl(final DemoStrategyFactoryProvider strategyFactoryProvider) {
        final DemoMoveFactory moveFactory = new DemoDefaultMoveFactory();

        final List<DemoStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final DemoStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Demo";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return DemoGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return DemoGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public DemoGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final DemoGameBuilder gameBuilder = new DemoGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(DemoGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(DemoGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final DemoPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final DemoStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final DemoPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final DemoStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Demo game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(
                        DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_NO,
                        "Player's outcome on No/No",
                        Optional.of(-1))
                .needInteger(
                        DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_YES,
                        "Player's outcome on No/Yes",
                        Optional.of(0))
                .needInteger(
                        DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_NO,
                        "Player's outcome on Yes/No",
                        Optional.of(0))
                .needInteger(
                        DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_YES,
                        "Player's outcome on Yes/Yes",
                        Optional.of(1))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Creates a Demo player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link DemoPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private DemoPlayer createPlayer(final DemoPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<Boolean, Map<Boolean, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<Boolean, Double> possibleOutcomesNo = new LinkedHashMap<>();
        possibleOutcomesNo.put(
                Boolean.FALSE,
                (double) (Integer) playerData.get(DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_NO));
        possibleOutcomesNo.put(
                Boolean.TRUE,
                (double) (Integer) playerData.get(DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_YES));
        possibleOutcomes.put(Boolean.FALSE, possibleOutcomesNo);

        final Map<Boolean, Double> possibleOutcomesYes = new LinkedHashMap<>();
        possibleOutcomesYes.put(
                Boolean.FALSE,
                (double) (Integer) playerData.get(DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_NO));
        possibleOutcomesYes.put(
                Boolean.TRUE,
                (double) (Integer) playerData.get(DemoGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_YES));
        possibleOutcomes.put(Boolean.TRUE, possibleOutcomesYes);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a Demo strategy.
     *
     * @param playerData The requested player data.
     * @return The Demo strategy.
     */
    private DemoStrategy getStrategy(final Map<String, Object> playerData) {
        return (DemoStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
