package de.fhdw.gaming.ipspiel22.vierGewinnt.gui.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGFieldState;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.event.ActionEvent;
import javafx.geometry.Point2D;
import javafx.geometry.Rectangle2D;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.layout.Region;
import javafx.scene.paint.Color;
import javafx.util.Duration;

/**
 * Displays a connect-4 field.
 */
public class VierGewinntFieldView extends Region {

    /**
     * Number of frames per second in an animation.
     */
    private static final int FRAMES_PER_SECOND = 25;
    /**
     * Fraction of the duration of the token change animation in relation to the duration of a move.
     */
    private static final double TOKEN_ANIMATION_RATIO = 0.5;
    /**
     * The field state.
     */
    private VGFieldState fieldState;
    /**
     * The canvas.
     */
    private final Canvas canvas;
    /**
     * The duration of a move.
     */
    private final ObjectProperty<Duration> moveDuration;
    /**
     * The current animation (if any).
     */
    private Optional<Timeline> currentAnimation;
    /**
     * {@code true} if the field is highlighted, else {@code false}.
     */
    private boolean highlighted;
  
    /**
     * Creates an {@link VierGewinntFieldView}.
     *
     * @param fieldState The field state.
     */
    VierGewinntFieldView(final VGFieldState fieldState) {
        this.fieldState = fieldState;
        this.moveDuration = new SimpleObjectProperty<>(Duration.millis(500.0));
        this.currentAnimation = Optional.empty();
        this.highlighted = false;

        this.canvas = new Canvas() {

            @Override
            public boolean isResizable() {
                return true;
            }

            @Override
            public double prefWidth(final double height) {
                return 0.0;
            }

            @Override
            public double prefHeight(final double width) {
                return 0.0;
            }

            @Override
            public double maxWidth(final double height) {
                return Double.POSITIVE_INFINITY;
            }

            @Override
            public double maxHeight(final double width) {
                return Double.POSITIVE_INFINITY;
            }
        };

        this.canvas.widthProperty().bind(this.widthProperty());
        this.canvas.heightProperty().bind(this.heightProperty());
        this.getChildren().add(this.canvas);

        this.widthProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.draw();
                });
        this.heightProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.draw();
                });

        this.setMinSize(50.0, 50.0);
        this.setMaxSize(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
    }

    /**
     * Returns the move duration property.
     */
    ObjectProperty<Duration> moveDurationProperty() {
        return this.moveDuration;
    }

    /**
     * Updates the state of the field control.
     *
     * @param fieldState The field state.
     */
    void setFieldState(final VGFieldState fieldState) {
        this.fieldState = fieldState;
        this.draw();
    }

    /**
     * Updates the highlight state of the field control.
     *
     * @param highlighted {@code true} if the field is highlighted, else {@code false}.
     */
    void setHighlighted(final boolean highlighted) {
        this.highlighted = highlighted;
        this.draw();
    }

    /**
     * Draws the field.
     */
    private void draw() {
        this.currentAnimation.ifPresent((final Timeline timeline) -> timeline.stop());

        final double size = this.getWidth();

        final GraphicsContext gc = this.canvas.getGraphicsContext2D();
        gc.setFill(this.highlighted ? Color.ROYALBLUE : Color.LIGHTSKYBLUE);
        gc.fillRect(0.0, 0.0, size, size);

        gc.setFill(Color.BLACK);
        gc.beginPath();
        gc.moveTo(0.0, 0.0);
        gc.lineTo(size, 0.0);
        gc.lineTo(size, size);
        gc.lineTo(0.0, size);
        gc.lineTo(0.0, 0.0);
        gc.closePath();
        gc.stroke();

        final double margin = size * 0.1;
        
        switch (this.fieldState) {
        case EMPTY:
            gc.setFill(Color.WHITE);
            gc.fillOval(margin, margin, size - 2 * margin, size - 2 * margin);
            break;
        case RED:
            gc.setFill(Color.WHITE);
            gc.fillOval(margin, margin, size - 2 * margin, size - 2 * margin);
            gc.setFill(Color.RED);
            this.animateTokenChange(gc, new Rectangle2D(margin, margin, size - 2 * margin, size - 2 * margin));
            break;
        case YELLOW:
            gc.setFill(Color.WHITE);
            gc.fillOval(margin, margin, size - 2 * margin, size - 2 * margin);
            gc.setFill(Color.YELLOW);
            this.animateTokenChange(gc, new Rectangle2D(margin, margin, size - 2 * margin, size - 2 * margin));
            break;
        default:
            throw new UnsupportedOperationException(
                    String.format("Unknown Connect 4 field state '%s'.", this.fieldState));
        }

    }

    /**
     * Animates a token change.
     *
     * @param gc   The {@link GraphicsContext} to use.
     * @param area The area where the token should be drawn.
     */
    private void animateTokenChange(final GraphicsContext gc, final Rectangle2D area) {
        final Point2D centre = new Point2D(
                area.getMinX() + area.getWidth() / 2.0,
                area.getMinY() + area.getHeight() / 2.0);

        final List<KeyFrame> keyFrames = new ArrayList<>();
        for (int keyFrameIndex = 1; keyFrameIndex < VierGewinntFieldView.FRAMES_PER_SECOND; ++keyFrameIndex) {
            final double diffX = area.getWidth() * keyFrameIndex / VierGewinntFieldView.FRAMES_PER_SECOND / 2;
            final double diffY = area.getHeight() * keyFrameIndex / VierGewinntFieldView.FRAMES_PER_SECOND / 2;

            final KeyFrame keyFrame = new KeyFrame(
                    this.moveDuration.get().multiply(VierGewinntFieldView.TOKEN_ANIMATION_RATIO).multiply(keyFrameIndex)
                            .divide(VierGewinntFieldView.FRAMES_PER_SECOND),
                    (final ActionEvent event) -> {
                        gc.fillOval(centre.getX() - diffX, centre.getY() - diffY, diffX * 2, diffY * 2);
                    });
            keyFrames.add(keyFrame);
        }

        final KeyFrame finalKeyFrame = new KeyFrame(
                this.moveDuration.get().multiply(VierGewinntFieldView.TOKEN_ANIMATION_RATIO),
                (final ActionEvent event) -> {
                    gc.fillOval(area.getMinX(), area.getMinY(), area.getWidth(), area.getHeight());
                });
        keyFrames.add(finalKeyFrame);

        this.currentAnimation = Optional.of(new Timeline(keyFrames.toArray(new KeyFrame[0])));
        this.currentAnimation.get().play();
    }
    

}
