/*
 * Copyright © 2020 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of VG-core.
 *
 * VG-core is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * VG-core is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VG-core. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel22.vierGewinnt.domain.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGAnswerEnum;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGBoard;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGField;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGFieldState;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPosition;

/**
 * Implements {@link VGBoard}.
 */
public final class VGBoardImpl implements VGBoard {

    /**
     * The number of columns.
     */
    private static final int BOARD_COLUMNS = 7;

    /**
     * The number of rows.
     */
    private static final int BOARD_ROWS = 6;

    /**
     * The fields of the board. Column by Column
     */
    private final List<List<VGFieldImpl>> fields;
    /**
     * The fields sorted by state.
     */
    private final Map<VGFieldState, Map<VGPosition, VGFieldImpl>> fieldsByState;

    /**
     * Creates an VG board.
     *
     * @throws IllegalArgumentException if the size does not meet the requirements.
     */
    public VGBoardImpl() throws IllegalArgumentException {
        final int rows = BOARD_ROWS;
        final int columns = BOARD_COLUMNS;

        this.fieldsByState = new LinkedHashMap<>();
        final Map<VGPosition, VGFieldImpl> emptyFields = new LinkedHashMap<>();
        this.fieldsByState.put(VGFieldState.EMPTY, emptyFields);
        this.fieldsByState.put(VGFieldState.RED, new LinkedHashMap<>());
        this.fieldsByState.put(VGFieldState.YELLOW, new LinkedHashMap<>());

        this.fields = new ArrayList<>(columns);
        for (int columnIndex = 0; columnIndex < columns; ++columnIndex) {
            final List<VGFieldImpl> column = new ArrayList<>(rows);
            for (int rowIndex = 0; rowIndex < rows; ++rowIndex) {
                final VGPosition position = VGPosition.of(columnIndex, rowIndex);
                final VGFieldImpl field = new VGFieldImpl(this, position, VGFieldState.EMPTY);
                column.add(field);
                emptyFields.put(position, field);
            }
            this.fields.add(column);
        }
    }

    /**
     * Copies an VG board.
     *
     * @param source The board to copy.
     */
    public VGBoardImpl(final VGBoardImpl source) {
        Objects.requireNonNull(source, "source");

        this.fieldsByState = new LinkedHashMap<>();
        this.fieldsByState.put(VGFieldState.EMPTY, new LinkedHashMap<>());
        this.fieldsByState.put(VGFieldState.RED, new LinkedHashMap<>());
        this.fieldsByState.put(VGFieldState.YELLOW, new LinkedHashMap<>());

        final int rows = source.getRows();
        final int columns = source.getColumns();
        this.fields = new ArrayList<>(columns);
        for (int columnIndex = 0; columnIndex < columns; ++columnIndex) {
            final List<VGFieldImpl> originColumn = source.fields.get(columnIndex);
            final List<VGFieldImpl> column = new ArrayList<>(rows);
            for (int rowIndex = 0; rowIndex < rows; ++rowIndex) {
                final VGFieldImpl originField = originColumn.get(rowIndex);
                final VGFieldImpl field = new VGFieldImpl(
                        this,
                        VGPosition.of(columnIndex, rowIndex),
                        originField.getState());
                column.add(field);
                this.fieldsByState.get(field.getState()).put(field.getPosition(), field);
            }
            this.fields.add(column);
        }
    }

    @Override
    public String toString() {
        return String.format("VGBoard[size=%d, fields=%s]", this.fields.size(), this.fields);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof VGBoardImpl) {
            final VGBoardImpl other = (VGBoardImpl) obj;
            return this.fields.equals(other.fields);
        }
        return false;
    }
    
    @Override
    public int hashCode() {
        return this.fields.hashCode();
    }

    @Override
    public boolean hasFieldAt(final VGPosition position) {
        final int rowSize = this.getRows();
        final int columnSize = this.getColumns();
        final int row = position.getRow();
        final int column = position.getColumn();
        return row >= 0 && row < rowSize && column >= 0 && column < columnSize;
    }

    @Override
    public VGField getFieldAt(final VGPosition position) {
        if (!this.hasFieldAt(position)) {
            throw new IllegalArgumentException(String.format("Position %s out of range.", position));
        }
        return this.fields.get(position.getColumn()).get(position.getRow());
    }

    @Override
    public List<List<? extends VGField>> getFields() {
        final List<List<? extends VGField>> result = new ArrayList<>();
        this.fields.forEach((final List<VGFieldImpl> column) -> result.add(new ArrayList<>(column)));
        return result;
    }

    @Override
    public Map<VGPosition, VGField> getFieldsBeing(final VGFieldState fieldState) {
        return Collections.unmodifiableMap(this.fieldsByState.get(fieldState));
    }

    @Override
    public VGBoardImpl deepCopy() {
        return new VGBoardImpl(this);
    }

    /**
     * This operation is called by a {@link VGFieldImpl} when a field changes its state.
     *
     * @param field    The field that changed its state.
     * @param oldState The old state of the field.
     */
    void fieldChangedState(final VGFieldImpl field, final VGFieldState oldState) {
        this.fieldsByState.get(oldState).remove(field.getPosition());
        this.fieldsByState.get(field.getState()).put(field.getPosition(), field);
    }

    @Override
    public int getRows() {
        return this.fields.get(0).size();
    }

    @Override
    public int getColumns() {
        return this.fields.size();
    }

    @Override
    public VGField getNextFieldInColumn(final VGAnswerEnum column) {
        if (column.equals(VGAnswerEnum.FIRSTCOLUMN)) {
            return getNextField(0);
        } else if (column.equals(VGAnswerEnum.SECONDCOLUMN)) {
            return getNextField(1);
        } else if (column.equals(VGAnswerEnum.THIRDCOLUMN)) {
            return getNextField(2);
        } else if (column.equals(VGAnswerEnum.FOURTHCOLUMN)) {
            return getNextField(3);
        } else if (column.equals(VGAnswerEnum.FITFHCOLUMN)) {
            return getNextField(4);
        } else if (column.equals(VGAnswerEnum.SIXTHCOLUMN)) {
            return getNextField(5);
        } else {
            return getNextField(6);
        }
    }

    /**
     * This operation is called in getNextFieldInColumn.
     * 
     * @param indexInList Index der Spalte der fields-Liste.
     */
    private VGField getNextField(final Integer indexInList) {
        for (final VGFieldImpl field : fields.get(indexInList)) {
            if (field.getState().equals(VGFieldState.EMPTY)) {
                return field;
            }
        }
        return null;
    }
}
