package de.fhdw.gaming.ipspiel22.kopfundzahl.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlGameBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlGameBuilderFactory;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlPlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlPlayerBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlStrategy;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.factory.KopfundZahlDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.factory.KopfundZahlStrategyFactory;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.factory.KopfundZahlStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.kopfundzahl.moves.factory.KopfundZahlMoveFactory;
import de.fhdw.gaming.ipspiel22.kopfundzahl.moves.impl.DefaultKopfundZahlMoveFactory;

/**
 * Implements {@link KopfundZahlGameBuilderFactory} by creating a KopfundZahl game builder.
 */
public final class KopfundZahlGameBuilderFactoryImpl implements KopfundZahlGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;

    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;

    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Head and Tail strategies.
     */
    private final Set<KopfundZahlStrategy> strategies;

    /**
     * Creates a Head and Tail game factory. Head and Tail strategies are loaded
     * by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public KopfundZahlGameBuilderFactoryImpl() {
        this(new KopfundZahlDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Head and Tail game factory.
     *
     * @param strategyFactoryProvider The {@link KopfundZahlStrategyFactoryProvider} for loading Head and
     *                                Tail strategies.
     */
    KopfundZahlGameBuilderFactoryImpl(final KopfundZahlDefaultStrategyFactoryProvider strategyFactoryProvider) {
        final KopfundZahlMoveFactory moveFactory = new DefaultKopfundZahlMoveFactory();

        final List<KopfundZahlStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final KopfundZahlStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "KopfundZahl";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return KopfundZahlGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return KopfundZahlGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public KopfundZahlGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final KopfundZahlGameBuilder gameBuilder = new KopfundZahlGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String, Object> gameData = inputProvider.needInteger(
                    GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                    "Maximum computation time per move in seconds",
                    Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MinValueValidator<>(KopfundZahlGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MaxValueValidator<>(KopfundZahlGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                    .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final KopfundZahlPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final KopfundZahlStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final KopfundZahlPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(),
                    secondPlayerData);
            final KopfundZahlStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating KopfundZahl game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {
        if (title.equals("Player 1")) {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_TAIL,
                            "Player's outcome on Tail/Tail",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_HEAD,
                            "Player's outcome on Tail/Head",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_TAIL,
                            "Player's outcome on Head/Tail",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_HEAD,
                            "Player's outcome on Head/Head",
                            Optional.of(1))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(),
                            this.strategies);

        } else {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_TAIL,
                            "Player's outcome on Tail/Tail",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_HEAD,
                            "Player's outcome on Tail/Head",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_TAIL,
                            "Player's outcome on Head/Tail",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_HEAD,
                            "Player's outcome on Head/Head",
                            Optional.of(-1))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(),
                            this.strategies);

        }
        return inputProvider.requestData(title);
    }

    /**
     * Creates a KopfundZahl player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link KopfundZahlPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private KopfundZahlPlayer createPlayer(final KopfundZahlPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<Boolean, Map<Boolean, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<Boolean, Double> possibleOutcomesNo = new LinkedHashMap<>();
        possibleOutcomesNo.put(
                Boolean.FALSE,
                (double) (Integer) playerData.get(KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_TAIL));
        possibleOutcomesNo.put(
                Boolean.TRUE,
                (double) (Integer) playerData.get(KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_HEAD));
        possibleOutcomes.put(Boolean.FALSE, possibleOutcomesNo);

        final Map<Boolean, Double> possibleOutcomesYes = new LinkedHashMap<>();
        possibleOutcomesYes.put(
                Boolean.FALSE,
                (double) (Integer) playerData.get(KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_TAIL));
        possibleOutcomesYes.put(
                Boolean.TRUE,
                (double) (Integer) playerData.get(KopfundZahlGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_HEAD));
        possibleOutcomes.put(Boolean.TRUE, possibleOutcomesYes);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a KopfundZahl strategy.
     *
     * @param playerData The requested player data.
     * @return The KopfundZahl strategy.
     */
    private KopfundZahlStrategy getStrategy(final Map<String, Object> playerData) {
        return (KopfundZahlStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }

}
