package de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteAnswerEnum;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayerBuilder;

/**
 * Implements {@link KopfundZahlundKantePlayerBuilder}.
 */
public final class KopfundZahlundKantePlayerBuilderImpl implements KopfundZahlundKantePlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<KopfundZahlundKanteAnswerEnum, Map<KopfundZahlundKanteAnswerEnum, Double>>> possibleOutcomes;

    /**
     * Creates an {@link KopfundZahlundKantePlayerBuilderImpl}.
     */
    public KopfundZahlundKantePlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public KopfundZahlundKantePlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public KopfundZahlundKantePlayerBuilder changePossibleOutcomes(final Map<KopfundZahlundKanteAnswerEnum,
            Map<KopfundZahlundKanteAnswerEnum,
                    Double>> newpossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newpossibleOutcomes);
        return this;
    }

    @Override
    public KopfundZahlundKantePlayer build() throws GameException {
        return new KopfundZahlundKantePlayerImpl(
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<KopfundZahlundKanteAnswerEnum, Map<KopfundZahlundKanteAnswerEnum, Double>> checkPossibleOutcomes(
            final Map<KopfundZahlundKanteAnswerEnum, Map<KopfundZahlundKanteAnswerEnum, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.TAIL, KopfundZahlundKanteAnswerEnum.HEAD);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.TAIL, KopfundZahlundKanteAnswerEnum.EDGE);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.TAIL, KopfundZahlundKanteAnswerEnum.TAIL);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.HEAD, KopfundZahlundKanteAnswerEnum.TAIL);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.HEAD, KopfundZahlundKanteAnswerEnum.HEAD);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.HEAD, KopfundZahlundKanteAnswerEnum.EDGE);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.EDGE, KopfundZahlundKanteAnswerEnum.HEAD);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.EDGE, KopfundZahlundKanteAnswerEnum.TAIL);
        this.checkPossibleOutcome(outcomes, KopfundZahlundKanteAnswerEnum.EDGE, KopfundZahlundKanteAnswerEnum.EDGE);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<KopfundZahlundKanteAnswerEnum, Map<KopfundZahlundKanteAnswerEnum,
            Double>> outcomes, final KopfundZahlundKanteAnswerEnum firstChoice,
            final KopfundZahlundKanteAnswerEnum secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "No outcome defined for player '%s' and combination %s/%s.",
                            this.name,
                            firstChoice,
                            secondChoice));
        }
    }

}
