package de.schereSteinPapier.domain.impl;

import static de.schereSteinPapier.SSPConstants.AuswahlConstants.PAPIER;
import static de.schereSteinPapier.SSPConstants.AuswahlConstants.SCHERE;
import static de.schereSteinPapier.SSPConstants.AuswahlConstants.STEIN;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.schereSteinPapier.domain.SSPGameBuilder;
import de.schereSteinPapier.domain.SSPGameBuilderFactory;
import de.schereSteinPapier.domain.SSPPlayer;
import de.schereSteinPapier.domain.SSPPlayerBuilder;
import de.schereSteinPapier.domain.SSPStrategy;
import de.schereSteinPapier.domain.factory.SSPDefaultStrategyFactoryProvider;
import de.schereSteinPapier.domain.factory.SSPStrategyFactory;
import de.schereSteinPapier.domain.factory.SSPStrategyFactoryProvider;
import de.schereSteinPapier.moves.factory.SSPMoveFactory;
import de.schereSteinPapier.moves.impl.SSPDefaultMoveFactory;

/**
 * Implements {@link SSPGameBuilderFactory} by creating a SSP game builder.
 */
public final class SSPGameBuilderFactoryImpl implements SSPGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available SSP strategies.
     */
    private final Set<SSPStrategy> strategies;

    /**
     * Creates a SSP game factory. SSP strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public SSPGameBuilderFactoryImpl() {
        this(new SSPDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a SSP game factory.
     *
     * @param strategyFactoryProvider The {@link SSPStrategyFactoryProvider} for loading SSP strategies.
     */
    SSPGameBuilderFactoryImpl(final SSPStrategyFactoryProvider strategyFactoryProvider) {
        final SSPMoveFactory moveFactory = new SSPDefaultMoveFactory();

        final List<SSPStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final SSPStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "SSP";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return SSPGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return SSPGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public SSPGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final SSPGameBuilder gameBuilder = new SSPGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(SSPGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(SSPGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final SSPPlayer firstPlayer = this
                    .createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final SSPStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final SSPPlayer secondPlayer = this
                    .createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final SSPStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Demo game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {
        if (title.equals("Player 1")) {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_SCHERE,
                            "Player's outcome on Schere/Schere",
                            Optional.of(0))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_STEIN,
                            "Player's outcome on Schere/Stein",
                            Optional.of(-1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_PAPIER,
                            "Player's outcome on Schere/Papier",
                            Optional.of(1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_SCHERE,
                            "Player's outcome on Stein/Schere",
                            Optional.of(1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_STEIN,
                            "Player's outcome on Stein/Stein",
                            Optional.of(0))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_PAPIER,
                            "Player's outcome on Stein/Papier",
                            Optional.of(-1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_SCHERE,
                            "Player's outcome on Papier/Schere",
                            Optional.of(-1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_STEIN,
                            "Player's outcome on Papier/Stein",
                            Optional.of(1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_PAPIER,
                            "Player's outcome on Papier/Papier",
                            Optional.of(0))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy",
                            Optional.empty(), this.strategies);
        } else {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_SCHERE,
                            "Player's outcome on Schere/Schere",
                            Optional.of(0))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_STEIN,
                            "Player's outcome on Schere/Stein",
                            Optional.of(1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_PAPIER,
                            "Player's outcome on Schere/Papier",
                            Optional.of(-1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_SCHERE,
                            "Player's outcome on Stein/Schere",
                            Optional.of(-1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_STEIN,
                            "Player's outcome on Stein/Stein",
                            Optional.of(0))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_PAPIER,
                            "Player's outcome on Stein/Papier",
                            Optional.of(1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_SCHERE,
                            "Player's outcome on Papier/Schere",
                            Optional.of(1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_STEIN,
                            "Player's outcome on Papier/Stein",
                            Optional.of(-1))
                    .needInteger(
                            SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_PAPIER,
                            "Player's outcome on Papier/Papier",
                            Optional.of(0))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy",
                            Optional.empty(), this.strategies);
        }
        return inputProvider.requestData(title);
    }

    /**
     * Creates a SSP player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link SSPPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private SSPPlayer createPlayer(final SSPPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<String, Map<String, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<String, Double> possibleOutcomesSchere = new LinkedHashMap<>();
        possibleOutcomesSchere.put(
                SCHERE,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_SCHERE));
        possibleOutcomesSchere.put(
                STEIN,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_STEIN));
        possibleOutcomesSchere.put(
                PAPIER,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCHERE_PAPIER));
        possibleOutcomes.put(SCHERE, possibleOutcomesSchere);

        final Map<String, Double> possibleOutcomesStein = new LinkedHashMap<>();
        possibleOutcomesStein.put(
                SCHERE,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_SCHERE));
        possibleOutcomesStein.put(
                STEIN,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_STEIN));
        possibleOutcomesStein.put(
                PAPIER,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STEIN_PAPIER));
        possibleOutcomes.put(STEIN, possibleOutcomesStein);

        final Map<String, Double> possibleOutcomesPapier = new LinkedHashMap<>();
        possibleOutcomesPapier.put(
                SCHERE,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_SCHERE));
        possibleOutcomesPapier.put(
                STEIN,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_STEIN));
        possibleOutcomesPapier.put(
                PAPIER,
                (double) (Integer) playerData.get(SSPGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPIER_PAPIER));
        possibleOutcomes.put(PAPIER, possibleOutcomesPapier);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a SSP strategy.
     *
     * @param playerData The requested player data.
     * @return The SSP strategy.
     */
    private SSPStrategy getStrategy(final Map<String, Object> playerData) {
        return (SSPStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
