package de.schereSteinPapier.strategy;

import java.util.Optional;
import java.util.Random;

import de.schereSteinPapier.domain.SSPPlayer;
import de.schereSteinPapier.domain.SSPState;
import de.schereSteinPapier.domain.SSPStrategy;
import de.schereSteinPapier.moves.SSPMove;
import de.schereSteinPapier.moves.factory.SSPMoveFactory;

/**
 * Implements {@link SSPStrategy} by saying "Papier, Stein, Schere" with a 1/3 chance per entry.
 */
public class SSPSayEquilibriumStrategy implements SSPStrategy {

    /**
     * The random number generator.
     */
    private static final Random RANDOM = new Random();

    /**
     * The factory for creating SSP moves.
     */
    private final SSPMoveFactory moveFactory;

    /**
     * Creates an {@link SSPSayEquilibriumStrategy}.
     *
     * @param moveFactory The factory for creating SSP moves.
     */
    SSPSayEquilibriumStrategy(final SSPMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    /**
     * returns "SCHERE", "STEIN", "PAPIER" with a chance of 1/3 each.
     */
    public Optional<SSPMove> weightedRandomAnswer() {
        // generates a random number between 0 and 2 inclusively
        final int weight = RANDOM.nextInt(3);
        if (weight < 1) {
            return Optional.of(this.moveFactory.createSchereMove());
        }
        if (weight < 2) {
            return Optional.of(this.moveFactory.createSteinMove());
        }
        return Optional.of(this.moveFactory.createPapierMove());
    }

    @Override
    public Optional<SSPMove> computeNextMove(final int gameId, final SSPPlayer player, final SSPState state) {
        return this.weightedRandomAnswer();
    }

    @Override
    public String toString() {
        return SSPSayEquilibriumStrategy.class.getSimpleName();
    }
}
