package de.fhdw.gaming.ipspiel22.vierGewinnt.moves.impl;

import java.util.Objects;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGAnswerEnum;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGFieldState;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPlayer;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPosition;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGState;

/**
 * Represents a move which places a token on the board.
 * <p>
 * Note that this move is only allowed if the colour of the token placed belongs to the corresponding player, and if the
 * move leads to at least one other token to be flipped.
 */
final class VGPlaceTokenMove extends AbstractVGMove {

    /**
     * {@code true} if a red token is placed, and {@code false} if a yellow token is placed.
     */
    private final boolean placingRedToken;
    /**
     * The position of the token placed on the board.
     */
    private final VGPosition tokenPosition;

    /**
     * Creates a {@link VGPlaceTokenMove} object.
     * 
     * @param placingRedToken {@code true} if a red token is placed, and {@code false} if a yellow token is placed.
     * @param tokenPosition   The position of the token placed on the board.
     */
    VGPlaceTokenMove(final boolean placingRedToken, final VGPosition tokenPosition) {
        this.placingRedToken = placingRedToken;
        this.tokenPosition = Objects.requireNonNull(tokenPosition, "tokenPosition");
    }

    /**
     * Returns {@code true} if a red token is placed, and {@code false} if a yellow token is placed.
     */
    boolean isPlacingRedToken() {
        return this.placingRedToken;
    }

    /**
     * Returns the position of the token placed on the board.
     */
    VGPosition getTokenPosition() {
        return this.tokenPosition;
    }

    @Override
    public void applyTo(final VGState state, final VGPlayer player) throws GameException {
        if (this.isPlacingRedToken() != player.isUsingRedChips()) {
            throw new GameException(
                    String.format(
                            "Player %s cannot place a %s token.",
                            player,
                            this.isPlacingRedToken() ? VGFieldState.RED : VGFieldState.YELLOW));
        }

        state.getBoard().getNextFieldInColumn(this.toAnswerEnum()).placeToken(player.isUsingRedChips());
    }

    /**
     * Translate an integer from 1 to 7 into a VGAnswerEnum.
     * 
     * @return VGAnswerEnum.
     */
    private VGAnswerEnum toAnswerEnum() {
        if (this.tokenPosition.getColumn() == 0) {
            return VGAnswerEnum.FIRSTCOLUMN;
        } else if (this.tokenPosition.getColumn() == 1) {
            return VGAnswerEnum.SECONDCOLUMN;
        } else if (this.tokenPosition.getColumn() == 2) {
            return VGAnswerEnum.THIRDCOLUMN;
        } else if (this.tokenPosition.getColumn() == 3) {
            return VGAnswerEnum.FOURTHCOLUMN;
        } else if (this.tokenPosition.getColumn() == 4) {
            return VGAnswerEnum.FITFHCOLUMN;
        } else if (this.tokenPosition.getColumn() == 5) {
            return VGAnswerEnum.SIXTHCOLUMN;
        } else {
            return VGAnswerEnum.SEVENTHCOLUMN;
        }
    }

    @Override
    public String toString() {
        return String.format(
                "Placing %s token on field at %s",
                this.placingRedToken ? "red" : "yellow",
                this.tokenPosition);
    }

    @Override
    public int getColumnInt() {
        return 0;
    }

}
