package de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDGame;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDGameBuilder;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDPlayer;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDPlayerBuilder;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDStrategy;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.moves.impl.AbstractGDMove;

/**
 * Implements {@link GDGameBuilder}.
 */
public class GDGameBuilderImpl implements GDGameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;

    /**
     * The player using black tokens.
     */
    private Optional<GDPlayer> firstPlayer;
    /**
     * The strategy of the player using black tokens.
     */
    private Optional<GDStrategy> firstPlayerStrategy;
    /**
     * The player using white tokens.
     */
    private Optional<GDPlayer> secondPlayer;
    /**
     * The strategy of the player using white tokens.
     */
    private Optional<GDStrategy> secondPlayerStrategy;
    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a Gefangenen Dilemma game builder.
     */
    GDGameBuilderImpl() {
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public GDPlayerBuilder createPlayerBuilder() {
        return new GDPlayerBuilderImpl();
    }

    @Override
    public GDGameBuilder addPlayer(final GDPlayer player, final GDStrategy strategy)
            throws GameException {

        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public GDGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public GDGameBuilder changeObserverFactoryProvider(final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public GDGame build(final int id) throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("A Gefangenen Dilemma game needs two players.");
        }

        final GDStateImpl initialState = new GDStateImpl(this.firstPlayer.get(), this.secondPlayer.get());

        final Map<String, GDStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new GDGameImpl(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractGDMove.class::isInstance,
                this.observerFactoryProvider);
    }
    
    /**
     * Returns ObserverFactoryProvider.
     * @return
     */
    public ObserverFactoryProvider getObserverFactoryProvider() {
        return observerFactoryProvider;
    }

    /**
     * Returns maxComputationTimePerMove.
     * @return
     */
    public int getMaxComputationTimePerMove() {
        return maxComputationTimePerMove;
    }
}
