package de.fhdw.gaming.ipspiel23.c4.gui.impl;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.core.domain.State;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Game;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Observer;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;
import javafx.animation.Animation;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.geometry.VPos;
import javafx.scene.control.Label;
import javafx.scene.control.Slider;
import javafx.scene.layout.Background;
import javafx.scene.layout.BackgroundFill;
import javafx.scene.layout.Border;
import javafx.scene.layout.BorderStroke;
import javafx.scene.layout.BorderStrokeStyle;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.CornerRadii;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.RowConstraints;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import javafx.scene.text.Font;
import javafx.scene.text.FontPosture;
import javafx.scene.text.FontWeight;
import javafx.stage.Screen;
import javafx.util.Duration;

/**
 * Displays a C4-Board.
 */
@SuppressWarnings("PMD.TooManyFields")
public class C4BoardView implements IC4Observer {

    /**
     * The initial delay in seconds.
     */
    private static final double INITIAL_DELAY = 0.5;

    /**
     * The margin used by elements of the state pane.
     */
    private static final double STATE_PANE_MARGIN = 40.0;

    /**
     * Pattern for extracting the relevant parts of a strategy name.
     */
    private static final Pattern STRATEGY_NAME_PATTERN = Pattern.compile("^(Othello)?(.*?)(Strategy)?$");

    /**
     * The factor to multiply pixels with to receive points.
     */
    private final double pixelsToPointsFactor;

    /**
     * The field controls.
     */
    private final Map<IC4Position, C4FieldView> controls;

    /**
     * The pane containing everything.
     */
    private final HBox rootPane;

    /**
     * The pane containing the board and its border.
     */
    private final VBox boardPane;

    /**
     * The grid pane containing the fields.
     */
    private final GridPane fieldPane;

    /**
     * The label describing the current game state.
     */
    private Label gameStateDescription;

    /**
     * The animation of the current game state description (if any).
     */
    private Optional<Timeline> gameStateDescriptionAnimation;

    /**
     * The delay in milliseconds.
     */
    private final AtomicInteger delay;

    /**
     * The size of a field control.
     */
    private final SimpleDoubleProperty fieldControlSize;

    /**
     * The size of various margins.
     */
    private final SimpleDoubleProperty margin;

    /**
     * The padding used for GridPanes.
     */
    private final SimpleObjectProperty<Insets> gridPadding;

    /**
     * The font size in pixels.
     */
    private final SimpleDoubleProperty fontSizeInPixels;

    /**
     * The font used for border labels.
     */
    private final SimpleObjectProperty<Font> borderLabelFont;

    /**
     * The font used for labels texts in the player's pane.
     */
    private final SimpleObjectProperty<Font> labelTextFont;

    /**
     * The font used for label values in the player's pane.
     */
    private final SimpleObjectProperty<Font> labelValueFont;

    /**
     * The font used for displaying the game result.
     */
    private final SimpleObjectProperty<Font> gameResultFont;

    /**
     * The pane for the top edge.
     */
    private HBox topEdge;

    /**
     * The pane for the left edge.
     */
    private VBox leftEdge;

    /**
     * The pane for the right edge.
     */
    private VBox rightEdge;

    /**
     * The pane for the bottom edge.
     */
    private HBox bottomEdge;

    /**
     * The last game state.
     */
    private Optional<IC4State> lastGameState;

    /**
     * The {@link Semaphore} used by {@link C4BoardEventProviderImpl}.
     */
    private final Semaphore semaphore = new Semaphore(0);
    
    /**
     * Creates an {@link C4BoardView}.
     *
     * @param game The game.
     */
    C4BoardView(final IC4Game game) {
        this.pixelsToPointsFactor = 72.0 / Screen.getPrimary().getDpi();
        this.borderLabelFont = new SimpleObjectProperty<>(Font.getDefault());
        this.labelTextFont = new SimpleObjectProperty<>(Font.getDefault());
        this.labelValueFont = new SimpleObjectProperty<>(Font.getDefault());
        this.gameResultFont = new SimpleObjectProperty<>(Font.getDefault());

        this.fieldControlSize = new SimpleDoubleProperty(0.0);
        this.margin = new SimpleDoubleProperty(0.0);
        this.gridPadding = new SimpleObjectProperty<>(Insets.EMPTY);
        this.fontSizeInPixels = new SimpleDoubleProperty(0.0);

        this.controls = new LinkedHashMap<>();
        this.fieldPane = new GridPane();
        this.boardPane = this.createFieldWithBorderPane();

        final GridPane statePane = this.createStatePane(game);

        this.rootPane = new HBox();
        this.rootPane.getChildren().addAll(this.boardPane, statePane);
        HBox.setHgrow(this.boardPane, Priority.ALWAYS);
        HBox.setHgrow(statePane, Priority.ALWAYS);
        HBox.setMargin(statePane, new Insets(C4BoardView.STATE_PANE_MARGIN));

        this.delay = new AtomicInteger((int) (C4BoardView.INITIAL_DELAY * 1000.0));
        this.lastGameState = Optional.empty();
        game.addObserver(this);

        this.gameStateDescriptionAnimation = Optional.empty();
    }

    /**
     * Creates the pane displaying the board and its border.
     * <p>
     * Requires {@link #fieldControlSize} to be valid.
     */
    private VBox createFieldWithBorderPane() {
        final Background background = new Background(
                new BackgroundFill(Color.SANDYBROWN, CornerRadii.EMPTY, Insets.EMPTY));

        final HBox topLeftCorner = new HBox();
        topLeftCorner.setBackground(background);
        topLeftCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        topLeftCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.topEdge = new HBox();
        this.topEdge.setBackground(background);
        this.topEdge.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        this.topEdge.setAlignment(Pos.CENTER);
        final HBox topRightCorner = new HBox();
        topRightCorner.setBackground(background);
        topRightCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        topRightCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));

        HBox.setHgrow(topLeftCorner, Priority.NEVER);
        HBox.setHgrow(this.topEdge, Priority.NEVER);
        HBox.setHgrow(topRightCorner, Priority.NEVER);

        this.leftEdge = new VBox();
        this.leftEdge.setBackground(background);
        this.leftEdge.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.leftEdge.setAlignment(Pos.CENTER);
        this.rightEdge = new VBox();
        this.rightEdge.setBackground(background);
        this.rightEdge.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.rightEdge.setAlignment(Pos.CENTER);

        HBox.setHgrow(this.leftEdge, Priority.NEVER);
        HBox.setHgrow(this.fieldPane, Priority.NEVER);
        HBox.setHgrow(this.rightEdge, Priority.NEVER);

        final HBox bottomLeftCorner = new HBox();
        bottomLeftCorner.setBackground(background);
        bottomLeftCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        bottomLeftCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.bottomEdge = new HBox();
        this.bottomEdge.setBackground(background);
        this.bottomEdge.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        this.bottomEdge.setAlignment(Pos.CENTER);
        final HBox bottomRightCorner = new HBox();
        bottomRightCorner.setBackground(background);
        bottomRightCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        bottomRightCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));

        HBox.setHgrow(bottomLeftCorner, Priority.NEVER);
        HBox.setHgrow(this.bottomEdge, Priority.NEVER);
        HBox.setHgrow(bottomRightCorner, Priority.NEVER);

        final HBox top = new HBox();
        top.getChildren().addAll(topLeftCorner, this.topEdge, topRightCorner);
        final HBox centre = new HBox();
        centre.getChildren().addAll(this.leftEdge, this.fieldPane, this.rightEdge);
        final HBox bottom = new HBox();
        bottom.getChildren().addAll(bottomLeftCorner, this.bottomEdge, bottomRightCorner);

        VBox.setVgrow(top, Priority.NEVER);
        VBox.setVgrow(centre, Priority.NEVER);
        VBox.setVgrow(bottom, Priority.NEVER);

        final VBox vbox = new VBox();
        vbox.getChildren().addAll(top, centre, bottom);

        return vbox;
    }

    /**
     * Creates the pane displaying the game state.
     *
     * @param game The game.
     */
    private GridPane createStatePane(final IC4Game game) {
        // can't pick a specific player, so have to get them in a different way. 
        final List<IC4Player> players = game.getState().getPlayers().values().stream().toList();
        final GridPane firstPlayer = this.createPlayerPane(game, players.get(0));
        final GridPane secondPlayer = this.createPlayerPane(game, players.get(1));

        this.gameStateDescription = new Label();
        this.gameStateDescription.fontProperty().bind(this.gameResultFont);
        this.gameStateDescription.setTextFill(Color.RED);

        final Label delayLabel = new Label("Delay in seconds: ");

        final Slider delaySlider = new Slider(0.0, 5.0, C4BoardView.INITIAL_DELAY);
        delaySlider.setMajorTickUnit(C4BoardView.INITIAL_DELAY);
        delaySlider.setMinorTickCount(4);
        delaySlider.setBlockIncrement(C4BoardView.INITIAL_DELAY);
        delaySlider.setShowTickMarks(true);
        delaySlider.setShowTickLabels(true);
        delaySlider.snapToTicksProperty().set(true);
        delaySlider.valueProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.delay.set((int) (newValue.doubleValue() * 1000.0));
                });

        final HBox delayBox = new HBox(delayLabel, delaySlider);
        HBox.setHgrow(delayLabel, Priority.NEVER);
        HBox.setHgrow(delaySlider, Priority.ALWAYS);

        final GridPane gridPane = new GridPane();
        gridPane.vgapProperty().bind(this.margin.multiply(2.0));

        gridPane.addRow(0, firstPlayer);
        GridPane.setHalignment(firstPlayer, HPos.CENTER);
        GridPane.setHgrow(firstPlayer, Priority.ALWAYS);
        GridPane.setVgrow(firstPlayer, Priority.NEVER);

        gridPane.addRow(1, secondPlayer);
        GridPane.setHalignment(secondPlayer, HPos.CENTER);
        GridPane.setHgrow(secondPlayer, Priority.ALWAYS);
        GridPane.setVgrow(secondPlayer, Priority.NEVER);

        gridPane.addRow(2, this.gameStateDescription);
        GridPane.setHalignment(this.gameStateDescription, HPos.CENTER);
        GridPane.setHgrow(this.gameStateDescription, Priority.ALWAYS);
        GridPane.setVgrow(this.gameStateDescription, Priority.NEVER);

        gridPane.addRow(3, delayBox);
        GridPane.setHalignment(delayBox, HPos.CENTER);
        GridPane.setHgrow(delayBox, Priority.ALWAYS);
        GridPane.setVgrow(delayBox, Priority.NEVER);

        return gridPane;
    }

    /**
     * Returns the root pane.
     */
    HBox getNode() {
        return this.rootPane;
    }

    /**
     * Returns the {@link Semaphore} used for user interaction.
     */
    Semaphore getUserInputSemaphore() {
        return this.semaphore;
    }

    /**
     * Maps a position to the corresponding {@link C4FieldView}.
     *
     * @param position The position.
     * @return The field view (if it exists).
     */
    Optional<C4FieldView> getFieldView(final IC4Position position) {
        return Optional.ofNullable(this.controls.get(position));
    }

    /**
     * Sets the game state.
     *
     * @param text The game state.
     */
    private void setGameState(final String text) {
        this.gameStateDescriptionAnimation.ifPresent((final Timeline timeline) -> timeline.stop());
        this.gameStateDescription.setText(text);
        if (!text.isEmpty()) {
            this.gameStateDescriptionAnimation = Optional.of(
                    new Timeline(
                            new KeyFrame(Duration.seconds(0.5), evt -> this.gameStateDescription.setVisible(false)),
                            new KeyFrame(Duration.seconds(1.0), evt -> this.gameStateDescription.setVisible(true))));
            this.gameStateDescriptionAnimation.get().setCycleCount(Animation.INDEFINITE);
            this.gameStateDescriptionAnimation.get().play();
        }
    }

    /**
     * Called if the game has been paused.
     *
     * @param game The game.
     */
    void gamePaused(final IC4Game game) {
        Platform.runLater(() -> this.setGameState("P A U S E D"));
    }

    /**
     * Called if the game has been resumed.
     *
     * @param game The game.
     */
    void gameResumed(final IC4Game game) {
        Platform.runLater(() -> this.setGameState(""));
    }

    @Override
    public void started(final Game<?, ?, ?, ?> game, final State<?, ?> state) throws InterruptedException {
        Platform.runLater(() -> {
            this.lastGameState = Optional.of((IC4State) state);

            final IC4Board board = this.lastGameState.get().getBoard();
            final int numRows = board.getRowCount();
            final int numColumns = board.getColumnCount();

            this.addLabelsToTopAndButtom(numColumns);
            this.addLabelsToRightAndLeft(numRows);

            this.boardPane.widthProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.min(newValue.doubleValue(), this.boardPane.getHeight());
                        this.setBorderAndFieldSize(numRows, numColumns, size);
                    });
            this.boardPane.heightProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.min(newValue.doubleValue(), this.boardPane.getWidth());
                        this.setBorderAndFieldSize(numRows, numColumns, size);
                    });

            this.fieldPane.widthProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.min(newValue.doubleValue(), this.fieldPane.getHeight());
                        this.fieldControlSize.set(size / numColumns);
                    });
            this.fieldPane.heightProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.min(newValue.doubleValue(), this.fieldPane.getWidth());
                        this.fieldControlSize.set(size / numRows);
                    });

            IntStream.range(0, numRows).forEachOrdered((final int row) -> {
                IntStream.range(0, numColumns).forEachOrdered((final int column) -> {
                    final IC4Position position = IC4Position.create(row, column);
                    final Optional<IC4Player> fieldOccupier = board.getOccupyingPlayerOrDefault(position);
                    final C4FieldView fieldControl = new C4FieldView(fieldOccupier);
                    this.fieldPane.add(fieldControl, column, row);
                    this.controls.put(position, fieldControl);

                    fieldControl.prefWidthProperty().bind(this.fieldControlSize);
                    fieldControl.prefHeightProperty().bind(this.fieldControlSize);
                });
            });
        });

        this.delayNextMove();
    }

    /**
     * Adds the labels to the top and bottom edges.
     *
     * @param numColumns The number of columns of the board.
     */
    private void addLabelsToTopAndButtom(final int numColumns) {
        for (int i = 0; i < numColumns; i++) {
            final Label topLabel = new Label(String.valueOf(i));
            topLabel.fontProperty().bind(this.borderLabelFont);
            topLabel.setMaxWidth(Double.MAX_VALUE);
            topLabel.setAlignment(Pos.CENTER);
            this.topEdge.getChildren().add(topLabel);
            HBox.setHgrow(topLabel, Priority.ALWAYS);

            final Label bottomLabel = new Label(String.valueOf(i));
            bottomLabel.fontProperty().bind(this.borderLabelFont);
            bottomLabel.setMaxWidth(Double.MAX_VALUE);
            bottomLabel.setAlignment(Pos.CENTER);
            this.bottomEdge.getChildren().add(bottomLabel);
            HBox.setHgrow(bottomLabel, Priority.ALWAYS);
        }
    }

    /**
     * Adds the labels to the right and left edges.
     *
     * @param numRows The number of rows of the board.
     */
    private void addLabelsToRightAndLeft(final int numRows) {
        for (int i = 0; i < numRows; i++) {
            final Label leftLabel = new Label(String.valueOf(i));
            leftLabel.fontProperty().bind(this.borderLabelFont);
            leftLabel.setMaxHeight(Double.MAX_VALUE);
            leftLabel.setAlignment(Pos.CENTER);
            this.leftEdge.getChildren().add(leftLabel);
            VBox.setVgrow(leftLabel, Priority.ALWAYS);

            final Label rightLabel = new Label(String.valueOf(i));
            rightLabel.fontProperty().bind(this.borderLabelFont);
            rightLabel.setMaxHeight(Double.MAX_VALUE);
            rightLabel.setAlignment(Pos.CENTER);
            this.rightEdge.getChildren().add(rightLabel);
            VBox.setVgrow(rightLabel, Priority.ALWAYS);
        }
    }
    

    /**
     * Sets the size of the board's fields and border.
     *
     * @param numRows The number of rows.
     * @param numColumns The number of columns.
     * @param boardWidth The width (and height) of the board in pixels.
     */
    private void setBorderAndFieldSize(final int numRows, final int numColumns, final double boardWidth) {
        
        final double boardHeight = boardWidth * ((double) numRows / numColumns);
        
        this.fieldPane.setPrefSize(boardWidth, boardHeight);
        this.bottomEdge.setPrefWidth(boardWidth);
        
        this.topEdge.setPrefWidth(boardWidth);
        this.leftEdge.setPrefHeight(boardHeight);
        this.rightEdge.setPrefHeight(boardHeight);
        this.boardPane.setPrefWidth(boardWidth);
        
        this.fontSizeInPixels.set(this.fieldControlSize.get() * 0.5);
        final double fontSize = this.fontSizeInPixels.get() * this.pixelsToPointsFactor;

        final Font fontRegular = Font.font(null, FontWeight.NORMAL, FontPosture.REGULAR, fontSize);
        final Font fontBold = Font.font(null, FontWeight.BOLD, FontPosture.REGULAR, fontSize);

        this.borderLabelFont.set(fontRegular);
        this.labelTextFont.set(fontBold);
        this.labelValueFont.set(fontRegular);
        this.gameResultFont.set(fontBold);

        this.margin.set(this.fontSizeInPixels.get() * 0.25);
        this.gridPadding.set(new Insets(this.margin.get()));
    }

    @Override
    public void nextPlayersComputed(final Game<?, ?, ?, ?> game, final State<?, ?> state,
            final Set<? extends Player<?>> players) {
        // nothing to do
    }

    @Override
    public void illegalPlayerRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player) {
        // nothing to do
    }

    @Override
    public void legalMoveApplied(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Move<?, ?> move) throws InterruptedException {

        Platform.runLater(() -> {
            final IC4State c4State = (IC4State) state;

            this.updateFields(c4State);

            this.lastGameState = Optional.of(c4State);
        });

        this.delayNextMove();
    }

    /**
     * Updates the fields and the label for a given field state after a move.
     *
     * @param state The game state.
     */
    private void updateFields(final IC4State state) {
        final Set<IC4Position> positionSet = new HashSet<>();
        for (final IC4Position ic4Position : state.getBoard().getPositionsByPlayer(state.getCurrentPlayer())) {
            positionSet.add(ic4Position);
        }
        final Set<IC4Position> oldPositionSet = new HashSet<>();
        for (final IC4Position ic4Position : this.lastGameState.get().getBoard()
                .getPositionsByPlayer(state.getCurrentPlayer())) {
            oldPositionSet.add(ic4Position);
        }
        positionSet.removeAll(oldPositionSet);
        positionSet.forEach((final IC4Position position) -> this.controls.get(position)
                .setOccupier(Optional.of(state.getCurrentPlayer())));
        
    }

    @Override
    public void illegalMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Optional<Move<?, ?>> move, final String reason) {
        // nothing to do
    }

    @Override
    public void overdueMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Optional<Move<?, ?>> chosenMove) {
        // nothing to do
    }

    @Override
    public void playerResigned(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player) {
        // nothing to do
    }

    @Override
    public void playerOvertaken(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> overtakenPlayer,
            final Player<?> overtakingPlayer) {
        // nothing to do
    }

    @Override
    public void finished(final Game<?, ?, ?, ?> game, final State<?, ?> state) {
        Platform.runLater(() -> {
            final IC4State c4State = (IC4State) state;
            final Optional<IC4Player> winner = Optional.of(c4State.getPlayers().values().stream()
                    .filter(p -> p.getState().equals(PlayerState.WON)).collect(Collectors.toList()).get(0));
            if (winner.isPresent()) {
                this.setGameState(winner.get().getName() + " WINS!");
            } else {
                this.setGameState("DRAW");
            }
        });
    }

    /**
     * Destroys this object.
     *
     * @param game The associated game.
     */
    public void destroy(final IC4Game game) {
        this.gameStateDescriptionAnimation.ifPresent((final Timeline timeline) -> timeline.stop());
        this.semaphore.release();
        game.removeObserver(this);
    }

    /**
     * Delays the execution of the next move.
     */
    private void delayNextMove() throws InterruptedException {
        Thread.sleep(this.delay.get());
    }

    /**
     * Returns a pane displaying the state of one player.
     *
     * @param game   The game.
     * @param player The player.
     * @return The requested pane.
     */
    private GridPane createPlayerPane(final IC4Game game, final IC4Player player) {
        final Label nameLabel = new Label("Name:");
        nameLabel.fontProperty().bind(this.labelTextFont);

        final Label nameText = new Label(player.getName());
        nameText.fontProperty().bind(this.labelValueFont);

        final Label colourLabel = new Label("Colour");
        colourLabel.fontProperty().bind(this.labelTextFont);

        final Label colourText = new Label();
        colourText.fontProperty().bind(this.labelValueFont);
        if (player.getToken() == 1) {
            final Circle circle = new Circle();
            circle.radiusProperty().bind(this.fontSizeInPixels.multiply(0.2));
            circle.setFill(Color.RED);
            colourText.setGraphic(circle);
        } else {
            final Circle circle = new Circle();
            circle.radiusProperty().bind(this.fontSizeInPixels.multiply(0.2));
            circle.setFill(Color.YELLOW);
            colourText.setGraphic(circle);
        }

        final Label strategyLabel = new Label("Strategy:");
        strategyLabel.fontProperty().bind(this.labelTextFont);

        final Label strategyText;
        final String rawName = game.getStrategies().get(player.getName()).toString();
        final Matcher nameMatcher = C4BoardView.STRATEGY_NAME_PATTERN.matcher(rawName);
        if (nameMatcher.matches()) {
            strategyText = new Label(nameMatcher.group(2));
        } else {
            strategyText = new Label(rawName);
        }
        strategyText.fontProperty().bind(this.labelValueFont);

        final GridPane gridPane = new GridPane();

        final ColumnConstraints labelColumnConstraints = new ColumnConstraints();
        labelColumnConstraints.setHalignment(HPos.RIGHT);
        labelColumnConstraints.setHgrow(Priority.NEVER);
        final ColumnConstraints textColumnConstraints = new ColumnConstraints();
        textColumnConstraints.setHalignment(HPos.LEFT);
        textColumnConstraints.setHgrow(Priority.ALWAYS);
        textColumnConstraints.setMaxWidth(Double.MAX_VALUE);
        gridPane.getColumnConstraints().addAll(labelColumnConstraints, textColumnConstraints);

        final RowConstraints rowConstraints = new RowConstraints();
        rowConstraints.setValignment(VPos.TOP);
        gridPane.getRowConstraints().addAll(rowConstraints, rowConstraints, rowConstraints, rowConstraints);

        gridPane.addRow(0, nameLabel, nameText);
        gridPane.addRow(1, colourLabel, colourText);
        gridPane.addRow(2, strategyLabel, strategyText);

        gridPane.setBorder(
                new Border(
                        new BorderStroke(Color.BLACK, BorderStrokeStyle.SOLID, CornerRadii.EMPTY, BorderStroke.THICK)));

        gridPane.hgapProperty().bind(this.margin.multiply(2.0));
        gridPane.paddingProperty().bind(this.gridPadding);

        return gridPane;
    }
}
