package de.fhdw.gaming.ipspiel23.c4.gui.impl;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.gui.GuiObserver;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Game;
import de.fhdw.gaming.ipspiel23.c4.gui.C4BoardEventProvider;
import javafx.scene.Node;

/**
 * implements {@link GuiObserver}.
 */
public class C4GuiObserverImpl implements GuiObserver {

    /**
     * The primary (i.e. first created) {@link C4GuiObserverImpl} instance of this class.
     */
    private static final AtomicReference<WeakReference<C4GuiObserverImpl>> INSTANCE = new AtomicReference<>();
    
    /**
     * The {@link C4BoardView} objects per C4 game ID.
     */
    private final Map<Integer, C4BoardView> boardViews;
    
    /**
     * Creates an {@link C4GuiObserverImpl}.
     */
    C4GuiObserverImpl() {
        C4GuiObserverImpl.INSTANCE.compareAndSet(null, new WeakReference<>(this));
        this.boardViews = new LinkedHashMap<>();
    }
    
    /**
     * Returns a {@link C4BoardEventProvider} for a given game.
     *
     * @param gameId The game ID.
     * @return The {@link C4BoardEventProvider}.
     */
    static Optional<C4BoardEventProvider> getEventProvider(final int gameId) {
        final C4GuiObserverImpl instance = Optional.ofNullable(C4GuiObserverImpl.INSTANCE.get())
                .map(Reference::get).orElse(null);
        if (instance == null) {
            return Optional.empty();
        } else {
            final C4BoardView boardView = instance.boardViews.get(gameId);
            return boardView == null ? Optional.empty() : Optional.of(new C4BoardEventProviderImpl(boardView));
        }
    }

    
    @Override
    public Optional<Node> gameCreated(final Game<?, ?, ?, ?> game) {
        if (game instanceof IC4Game) {
            final C4BoardView boardView = new C4BoardView((IC4Game) game);
            this.boardViews.put(game.getId(), boardView);
            return Optional.of(boardView.getNode());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public void gamePaused(final Game<?, ?, ?, ?> game) {
        if (game instanceof IC4Game) {
            final C4BoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gamePaused((IC4Game) game);
            }
        }
    }

    @Override
    public void gameResumed(final Game<?, ?, ?, ?> game) {
        if (game instanceof IC4Game) {
            final C4BoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gameResumed((IC4Game) game);
            }
        }
    }

    @Override
    public void gameDestroyed(final Game<?, ?, ?, ?> game) {
        if (game instanceof IC4Game) {
            final C4BoardView boardView = this.boardViews.remove(game.getId());
            if (boardView != null) {
                boardView.destroy((IC4Game) game);
            }
        }
    }
}
