package de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgGame;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgGameBuilder;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayer;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayerBuilder;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgStrategy;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.move.impl.AbstractAnswer;

/**
 * Builds a Game, implements {@link FzgGameBuilder}.
 * 
 */
public class FzgGameBuilderImpl implements FzgGameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;
    /**
     * The first player.
     */
    private Optional<FzgPlayer> firstPlayer;
    /**
     * The strategy of the first player.
     */
    private Optional<FzgStrategy> firstPlayerStrategy;
    /**
     * The second player.
     */
    private Optional<FzgPlayer> secondPlayer;
    /**
     * The strategy of the second player.
     */
    private Optional<FzgStrategy> secondPlayerStrategy;
    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Constructor, creates a FzgGameBuilder.
     */
    public FzgGameBuilderImpl() {
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public FzgPlayerBuilder createPlayerBuilder() {
        return new FzgPlayerBuilderImpl();
    }

    @Override
    public FzgGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public FzgGameBuilder addPlayer(final FzgPlayer player, final FzgStrategy strategy) throws GameException {
        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are not allowed."));
        }
        return this;
    }

    @Override
    public FzgGameBuilder changeObserverFactoryProvider(final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public FzgGame build(final int id) throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("Freizeitgestaltung needs two players.");
        }

        final FzgStateImpl initialState = new FzgStateImpl(this.firstPlayer.get(), this.secondPlayer.get());

        final Map<String, FzgStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new FzgGameImpl(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractAnswer.class::isInstance,
                this.observerFactoryProvider);
    }
}
