package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Map;
import java.util.Optional;
import java.util.Random;

import de.fhdw.gaming.core.domain.DefaultGame;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Game;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4MoveChecker;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;
import de.fhdw.gaming.ipspiel23.c4.moves.IC4Move;
import de.fhdw.gaming.ipspiel23.c4.moves.factory.IC4MoveFactory;
import de.fhdw.gaming.ipspiel23.c4.moves.impl.C4DefaultMoveFactory;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;

/**
 * The default implementation of the {@link IC4Game} interface.
 */
public class C4Game extends DefaultGame<IC4Player, IC4State, IC4Move, IC4Strategy> implements IC4Game {

    /**
     * Your friendly neighborhood random number generator.
     */
    private static final Random RANDOM = new Random();

    /**
     * The move factory.
     */
    private final IC4MoveFactory moveFactory;

    /**
     * Creates a new connect four game.
     *
     * @param id                        The ID of this game.
     * @param initialState              The initial state of the game.
     * @param strategies                The players' strategies.
     * @param maxComputationTimePerMove The maximum computation time per move in seconds.
     * @param moveChecker               The move checker.
     * @param observerFactoryProvider   The {@link ObserverFactoryProvider}.
     * @throws IllegalArgumentException if the player sets do not match.
     * @throws InterruptedException     if creating the game has been interrupted.
     */
    public C4Game(final int id, final IC4State initialState, final Map<String, IC4Strategy> strategies, 
        final long maxComputationTimePerMove, final IC4MoveChecker moveChecker, 
        final ObserverFactoryProvider observerFactoryProvider)
        throws IllegalArgumentException, InterruptedException {

        super(id, initialState, strategies, maxComputationTimePerMove, moveChecker, observerFactoryProvider);
        this.moveFactory = new C4DefaultMoveFactory();
    }

    @Override
    public Optional<IC4Move> chooseRandomMove(final IC4Player player, final IC4State state) {
        final IC4Position[] positions = state.getBoard().getLegalPositions();
        if (positions.length == 0) {
            return Optional.empty();
        }
        final int index = RANDOM.nextInt(positions.length);
        final IC4Position position = positions[index];
        return Optional.of(this.moveFactory.createMove(player, position));
    }
}
