package de.fhdw.gaming.ipspiel23.c4.moves.impl;

import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4Position;
import de.fhdw.gaming.ipspiel23.c4.moves.IC4Move;

/**
 * The default implementation of {@link IC4Move}.
 */
final class C4Move implements IC4Move {
    
    /**
     * The player that performs this move.
     */
    // I don't think we actually need the player in here 
    // (but the framework kinda forces us to have it ¯\_(ツ)_/¯)
    // yeah idunno, we basically only ever need the player passed to us 
    // via applyTo, but we still do some sanity checks against this
    // player instance here, just in case there actually *is* a meaning
    // behind all of this :)
    private final IC4Player localPlayer;

    /**
     * The row at which the token is inserted.
     */
    private final int row;

    /**
     * The column at which the token is inserted.
     */
    private final int column;

    /**
     * The lazily initialized target position.
     */
    private IC4Position position;

    /**
     * Creates a new instance of {@link C4Move}.
     * 
     * @param player The player that performs this move.
     * @param rowIndex The row at which the token is inserted.
     * @param columnIndex The column at which the token is inserted.
     */
    C4Move(final IC4Player player, final int rowIndex, final int columnIndex) {
        this.localPlayer = player;
        this.row = rowIndex;
        this.column = columnIndex;
    }

    /**
     * Creates a new instance of {@link C4Move}.
     * 
     * @param player The player that performs this move.
     * @param position The position at which the token is inserted.
     */
    C4Move(final IC4Player player, final IC4Position position) {
        this(player, position.getRow(), position.getColumn());
        this.position = position;
    }

    /**
     * Gets the player that performs this move.
     * @return The player that performs this move.
     */
    IC4Player getPlayer() {
        return this.localPlayer;
    }

    @Override
    public IC4Position getTargetPosition() {
        if (this.position == null) {
            this.position = new C4Position(this.row, this.column);
        }
        return this.position;
    }

    @Override
    public void applyTo(final IC4State state, final IC4Player player) throws GameException {

        // quick sanity check, still not sure why player would be different from localPlayer
        // but it can't hurt to be sure I guess :P
        if (player.getToken() != this.localPlayer.getToken()) {
            throw new IllegalStateException(String
                .format("The specified player '%s' cannot perform move '%s', which was reserved for '%s'", 
                player, this.localPlayer, this.toString()));
        }

        final IC4Field field = state.getBoard().getField(this.getTargetPosition());
        final boolean success = field.trySetOccupyingPlayer(player, false);
        if (!success) {
            throw new GameException(String.format("The requested move '%s' could not be performed. "
            + "Did you attempt to place a token in the air? Was the target position already occupied?", 
            this.toString()));
        }
        System.out.println(state.getBoard().toString());
        state.onMoveCompleted();
    }

    @Override
    public String toString() {
        return String.format("Insert token at (%s,%s)", this.column, this.row);
    }
    
    @Override
    public boolean equals(final Object obj) {
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IC4Move)) {
            return false;
        }
        final IC4Move other = (IC4Move) obj;
        return this.getTargetPosition().equals(other.getTargetPosition());
    }
    
    @Override
    public int hashCode() {
        int hash = 7;
        hash = hash * 31 + row;
        hash = hash * 31 + column;
        hash = hash * 31 + localPlayer.getToken();
        return hash;
    }
}

