package de.fhdw.gaming.ipspiel23.dilemma.domain.internals;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaState;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMove;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.moves.internals.DilemmaDefaultMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundData;
import de.fhdw.gaming.ipspiel23.memory.GameMemoryIdentifier;
import de.fhdw.gaming.ipspiel23.memory.MemoryState;

import static de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGameBuilder.DEFAULT_OUTCOME_COOPERATE_COOPERATE;
import static de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGameBuilder.DEFAULT_OUTCOME_COOPERATE_DEFECT;
import static de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGameBuilder.DEFAULT_OUTCOME_DEFECT_COOPERATE;
import static de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGameBuilder.DEFAULT_OUTCOME_DEFECT_DEFECT;

/**
 * Implements {@link IDilemmaState}.
 */
public final class DilemmaState extends MemoryState<IDilemmaPlayer, IDilemmaState, DilemmaRoundData> 
    implements IDilemmaState {

    /**
     * The first player.
     */
    private final IDilemmaPlayer firstPlayer;
    /**
     * The second player.
     */
    private final IDilemmaPlayer secondPlayer;

    /**
     * Creates a Dilemma state.
     *
     * @param firstPlayer  The first player.
     * @param secondPlayer The second player.
     * @throws GameException if less than two players are found and if both players share the same name.
     */
    public DilemmaState(final IDilemmaPlayer firstPlayer, final IDilemmaPlayer secondPlayer)
            throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayer, "firstPlayer");
        this.secondPlayer = Objects.requireNonNull(secondPlayer, "secondPlayer");

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a Dilemma state by copying an existing one.
     *
     * @param source The state to copy.
     */
    DilemmaState(final DilemmaState source) {
        this.firstPlayer = source.firstPlayer.deepCopy();
        this.secondPlayer = source.secondPlayer.deepCopy();
    }

    /**
     * Returns the first player.
     */
    @Override
    public IDilemmaPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public IDilemmaPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("DilemmaState[firstPlayer=%s, secondPlayer=%s]",
            this.firstPlayer,
            this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof DilemmaState) {
            final DilemmaState other = (DilemmaState) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public IDilemmaState deepCopy() {
        return new DilemmaState(this);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, IDilemmaPlayer> getPlayers() {
        final Map<String, IDilemmaPlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    /**
     *  Collects every player who didn't chose a move yet.
     */
    @Override
    public Set<IDilemmaPlayer> computeNextPlayers() {
        final Set<IDilemmaPlayer> playersWithoutMove = new LinkedHashSet<>();
        if (this.firstPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.firstPlayer);
        }
        if (this.secondPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.secondPlayer);
        }
        return playersWithoutMove;
    }

    @Override
    public IDilemmaPlayer getOpponentOf(final IDilemmaPlayer player) {
        if (this.firstPlayer.getName().equals(player.getName())) {
            return this.secondPlayer;
        }
        if (this.secondPlayer.getName().equals(player.getName())) {
            return this.firstPlayer;
        }
        throw new IllegalArgumentException("Unknown player: " + player);
    }

    /**
     *  Changes the state of the game according to the players' answers.
     */
    @Override
    public void nextTurn() {
        final Set<IDilemmaPlayer> playersWithoutMove = this.computeNextPlayers();
        if (playersWithoutMove.isEmpty()) {
            final DilemmaAnswerType answerOfFirstPlayer = this.firstPlayer.getAnswer().orElseThrow();
            final DilemmaAnswerType answerOfSecondPlayer = this.secondPlayer.getAnswer().orElseThrow();

            final Double outcomeOfFirstPlayer = this.firstPlayer
                .getPossibleOutcomes()
                .get(answerOfFirstPlayer)
                .get(answerOfSecondPlayer);
            this.firstPlayer.setState(outcomeToState(outcomeOfFirstPlayer));
            this.firstPlayer.setOutcome(outcomeOfFirstPlayer);

            final Double outcomeOfSecondPlayer = this.secondPlayer
                .getPossibleOutcomes()
                .get(answerOfSecondPlayer)
                .get(answerOfFirstPlayer);
            this.secondPlayer.setState(outcomeToState(outcomeOfSecondPlayer));
            this.secondPlayer.setOutcome(outcomeOfSecondPlayer);

            // update memory as needed
            final IDilemmaMoveFactory moveFactory = new DilemmaDefaultMoveFactory();
            final IDilemmaMove player1Move = moveFactory.fromAnswer(answerOfFirstPlayer);
            final IDilemmaMove player2Move = moveFactory.fromAnswer(answerOfSecondPlayer);
            
            final DilemmaRoundData round = DilemmaRoundData
                .fromRaw(firstPlayer, player1Move, secondPlayer, player2Move);

            storeRoundData(GameMemoryIdentifier.of(firstPlayer, firstPlayer.getStrategy(), 
                secondPlayer.getStrategy()), round);
            storeRoundData(GameMemoryIdentifier.of(secondPlayer, secondPlayer.getStrategy(),
                firstPlayer.getStrategy()), round);
        }
    }

    /**
     * Computes a player state from an outcome.
     *
     * @param outcome The player's outcome.
     */
    private static PlayerState outcomeToState(final Double outcome) {
        final double[] predefinedOutcomes = {
            (double) DEFAULT_OUTCOME_COOPERATE_COOPERATE,
            (double) DEFAULT_OUTCOME_COOPERATE_DEFECT,
            (double) DEFAULT_OUTCOME_DEFECT_COOPERATE,
            (double) DEFAULT_OUTCOME_DEFECT_DEFECT
        };
        final PlayerState[] states = {
            PlayerState.WON,
            PlayerState.LOST,
            PlayerState.WON,
            PlayerState.LOST,
        };
        for (int i = 0; i < states.length; i++) {
            if (isWithinDelta(outcome, predefinedOutcomes[i], 0.01d)) {
                return states[i];
            }
        }
        throw new IllegalStateException("Unable to map unknown outcome to player state :C");
    }
    
    /**
     * determines whether the specified value is within the delta of the comparand.
     * @param value the value
     * @param comparand the comparand
     * @param delta the delta
     */
    private static boolean isWithinDelta(final double value, final double comparand, final double delta) {
        return Math.abs(value - comparand) < delta;
    }
}
