package de.fhdw.gaming.ipspiel23.c4.domain.impl.evaluation;

import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4SolutionSlim;

import static de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim.EMPTY_TOKEN;

/**
 * A {@link C4SolutionAnalyzer} that analyzes the board on the diagonal from the bottom left to the top right.
 */
public class C4SolutionAnalyzerDiagonalLeft extends C4SolutionAnalyzerDiagonal {

    /**
     * Creates a new {@link C4SolutionAnalyzerDiagonalLeft}.
     * @param board The board to analyze.
     */
    public C4SolutionAnalyzerDiagonalLeft(final C4BoardSlim board) {
        super(board, C4Direction.NORTH_EAST);
    }

    @Override
    public IC4SolutionSlim tryFindFirstSolution(final IC4SolutionSlim currentSolution, final boolean updateCache) {
        // "Assignment of parameter 'currentSolution' is not allowed." :P
        IC4SolutionSlim solution = currentSolution;
        
        int iteration = 0;
        // determine start row of column 0 (skip first targetCount rows)
        for (int startRow = targetCount(); solution == null && startRow < rowMax(); startRow++, iteration++) {
            if (mayContainSolution(iteration)) {
                solution = tryFindFirstSolution(startRow, 0, iteration, updateCache);
            }
        }
        // determine start column of last row (ignore last targetCount columns)
        // don't evaluate diagonal starting in bottom left twice (startCol = 1)
        for (int startCol = 1; solution == null && startCol < colMax() - targetCount(); startCol++, iteration++) {
            if (mayContainSolution(iteration)) {
                solution = tryFindFirstSolution(rowMax() - 1, startCol, iteration, updateCache);
            }
        }
        return solution;
    }
    
    /**
     * Lazily searches for the first solution on the board starting at the specified position.
     * @param startRow the row to start at
     * @param startCol the column to start at
     * @param iteration the current iteration of the search
     * @param updateCache whether to update the solution cache, preventing the same line to be checked again
     * @return The first solution on the board or null if no solution was found.
     */
    private IC4SolutionSlim tryFindFirstSolution(final int startRow, final int startCol,
            final int iteration, final boolean updateCache) {
        int count = 0;
        int lastToken = 0;
        // "ForLoopVariableCount: Too many control variables in the for statement" :P
        int row = startRow;
        int col = startCol;
        boolean isDiagonalFull = true;
        for (; row >= 0 && col < colMax(); row--, col++) {
            final int token = board().getTokenUnsafe(row, col);
            count = countConsecutivesBranchless(count, token, lastToken);
            if (count >= targetCount()) {
                return scanRemaining(token, row, col);
            }
            isDiagonalFull &= token != EMPTY_TOKEN;
            lastToken = token;
        }
        if (updateCache && isDiagonalFull) {
            noSolutionIn(iteration);
        }
        return null;
    }

    @Override
    public void findAllSolutions(final Set<IC4SolutionSlim> resultSet, final boolean updateCache) {

        int iteration = 0;
        // determine start row of column 0 (skip first targetCount rows)
        for (int startRow = targetCount(); startRow < rowMax(); startRow++, iteration++) {
            if (mayContainSolution(iteration)) {
                findAllSolutions(resultSet, startRow, 0, iteration, updateCache);
            }
        }
        // determine start column of last row (ignore last targetCount columns)
        // don't evaluate diagonal starting in bottom left twice (startCol = 1)
        for (int startCol = 1; startCol < colMax() - targetCount(); startCol++, iteration++) {
            if (mayContainSolution(iteration)) {
                findAllSolutions(resultSet, rowMax() - 1, startCol, iteration, updateCache);
            }
        }
    }
    
    /**
     * Eagerly searches for all solutions on the board starting at the specified position.
     * @param startRow the row to start at
     * @param startCol the column to start at
     * @param iteration the current iteration of the search
     * @param updateCache whether to update the solution cache, preventing the same line to be checked again
     * @param resultSet the set to which any solutions are added
     */
    private void findAllSolutions(final Set<IC4SolutionSlim> resultSet, final int startRow, 
            final int startCol, final int iteration, final boolean updateCache) {
        int count = 0;
        int lastToken = 0;
        int row = startRow;
        int col = startCol;
        boolean isDiagonalFull = true;
        boolean diagonalContainsSolution = false;
        for (; row >= 0 && col < colMax(); row--, col++) {
            final int token = board().getTokenUnsafe(row, col);
            count = countConsecutivesBranchless(count, token, lastToken);
            if (count >= targetCount()) {
                count = 0;

                final C4SolutionSlim solution = scanRemaining(token, row, col);
                resultSet.add(solution);
                diagonalContainsSolution = true;
                // skip to end of solution
                row = solution.getRowIndexEnd();
                col = solution.getColumnIndexEnd();
            }
            isDiagonalFull &= token != EMPTY_TOKEN;
            lastToken = token;
        }
        if (updateCache && isDiagonalFull && !diagonalContainsSolution) {
            noSolutionIn(iteration);
        }
    }
    
    @Override
    protected C4SolutionSlim scanRemaining(final int token, final int startRow, final int startCol) {
        int row = startRow - 1;
        int col = startCol + 1;
        while (row >= 0 && col < colMax() && board().getTokenUnsafe(row, col) == token) {
            row--; 
            col++;
        }
        // revert last change to get inclusive upper bound
        row++;
        col--;
        // determine true solution size
        return solutionOf(token, row, col, board().getMinimumSolutionSize() + col - startCol);
    }
}
