package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;

/**
 * The default implementation of {@link IC4SolutionSlim}.
 */
public class C4SolutionSlim implements IC4SolutionSlim {
    
    /**
     * The token of the player who owns this solution.
     */
    private final int token;

    /**
     * The row index of the start position.
     */
    private final int rowIndexStart;

    /**
     * The column index of the start position.
     */
    private final int columnIndexStart;

    /**
     * The row index of the end position.
     */
    private final int rowIndexEnd;

    /**
     * The column index of the end position.
     */
    private final int columnIndexEnd;

    /**
     * The parent board that contains this solution.
     */
    private final C4BoardSlim parentBoard;

    /**
     * The direction of this solution.
     */
    private final C4Direction direction;

    /**
     * The number of fields in this solution.
     */
    private final int solutionSize;
    
    /**
     * The lazily initialized start position.
     */
    private IC4Position startPosition;

    /**
     * The lazily initialized end position.
     */
    private IC4Position endPosition;
    
    /**
     * Creates a new solution.
     * 
     * @param parentBoard The parent board that contains this solution.
     * @param token The token of the player who owns this solution.
     * @param rowIndexEnd The row index of the end position.
     * @param columnIndexEnd The column index of the end position.
     * @param direction The direction of this solution, in the direction from the start position to the end position.
     * @param solutionSize The number of fields in this solution.
     */
    public C4SolutionSlim(final C4BoardSlim parentBoard, final int token, final int rowIndexEnd, 
            final int columnIndexEnd, final C4Direction direction, final int solutionSize) {
        this.parentBoard = parentBoard;
        this.solutionSize = solutionSize;
        this.token = token;
        this.rowIndexEnd = rowIndexEnd;
        this.columnIndexEnd = columnIndexEnd;
        this.direction = direction;
        
        final C4Direction inverseDirection = direction.getInverse();
        // -1 because the end position is already included in the solution size
        final int steps = solutionSize - 1;
        this.rowIndexStart = inverseDirection.stepFromRow(rowIndexEnd, steps);
        this.columnIndexStart = inverseDirection.stepFromColumn(columnIndexEnd, steps);
    }

    /**
     * Returns the raw column end index, without allocating a new position.
     */
    public int getColumnIndexEnd() {
        return columnIndexEnd;
    }

    /**
     * Returns the raw row end index, without allocating a new position.
     */
    public int getRowIndexEnd() {
        return rowIndexEnd;
    }

    @Override
    public IC4Player getOwner() {
        return this.parentBoard.getPlayerByToken(token);
    }

    @Override
    public IC4Position getStartPosition() {
        // lazy heap allocation
        if (this.startPosition == null) {
            this.startPosition = new C4Position(this.rowIndexStart, this.columnIndexStart);
        }
        return this.startPosition;
    }

    @Override
    public IC4Position getEndPosition() {
        // lazy heap allocation
        if (this.endPosition == null) {
            this.endPosition = new C4Position(this.rowIndexEnd, this.columnIndexEnd);
        }
        return this.endPosition;
    }

    @Override
    public int size() {
        return this.solutionSize;
    }

    @Override
    public C4Direction getDirection() {
        return this.direction;
    }

    @Override
    public boolean equals(final Object other) {
        if (other == null) {
            return false;
        }
        if (!(other instanceof IC4SolutionSlim)) {
            return false;
        }
        final IC4SolutionSlim otherSolution = (IC4SolutionSlim) other;
        return this.token == otherSolution.getOwner().getToken() 
            // we don't care what direction a solution is facing.
            // the direction itself should probably be internal, but I guess
            // it may have valid use cases for consumers of our library :)
            && (this.getStartPosition().equals(otherSolution.getStartPosition()) 
                    && this.getEndPosition().equals(otherSolution.getEndPosition())
                || this.getStartPosition().equals(otherSolution.getEndPosition()) 
                    && this.getEndPosition().equals(otherSolution.getStartPosition()))
            && this.size() == otherSolution.size();
    }

    @Override
    public int hashCode() {
        int hash = 7 * 31;
        hash += this.token;
        hash = hash * 31 + (this.getStartPosition().hashCode() ^ this.getEndPosition().hashCode());
        hash = hash * 31 + this.size();
        return hash;
    }

    @Override
    public String toString() {
        return String.format("C4SolutionSlim: (%d, %d) -- %d -- (%d, %d)",
            this.rowIndexStart, this.columnIndexStart, this.token, this.rowIndexEnd, this.columnIndexEnd);
    }
}
