package de.fhdw.gaming.ipspiel23.c4.domain;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;

/**
 * A builder for configuring a connect four game.
 */
public interface IC4GameBuilder extends GameBuilder {

    /**
     * The default number of rows of the board.
     */
    int DEFAULT_BOARD_ROWS = 6;

    /**
     * The default number of columns of the board.
     */
    int DEFAULT_BOARD_COLUMNS = 7;

    /**
     * The default number of fields that are required for a solution.
     */
    int DEFAULT_REQUIRED_SOLUTION_SIZE = 4;

    /**
     * The default number of players.
     */
    int DEFAULT_PLAYER_COUNT = 2;

    /**
     * The default observer factory provider.
     */
    IC4PlayerBuilder createPlayerBuilder();

    /**
     * Adds a player to the game.
     * 
     * @param player the player to add
     * @param strategy the strategy used by the player
     * @return the same builder instance for fluent configuration
     */
    IC4GameBuilder addPlayer(IC4Player player, IC4Strategy strategy) throws GameException;

    /**
     * Changes the number of rows of the board.
     * 
     * @param newRowCount the new number of rows
     * @return the same builder instance for fluent configuration
     */
    IC4GameBuilder changeBoardRows(int newRowCount);

    /**
     * Changes the number of columns of the board.
     * 
     * @param newColumnCount the new number of columns
     * @return the same builder instance for fluent configuration
     */
    IC4GameBuilder changeBoardColumns(int newColumnCount);

    /**
     * Changes the number of fields that are required for a solution.
     * 
     * @param newSolutionSize the new number of fields that are required for a solution
     * @return the same builder instance for fluent configuration
     */
    IC4GameBuilder changeRequiredSolutionSize(int newSolutionSize);

    /**
     * Changes the observer factory provider.
     * 
     * @param newObserverFactoryProvider the new observer factory provider
     * @return the same builder instance for fluent configuration
     */
    IC4GameBuilder changeObserverFactoryProvider(ObserverFactoryProvider newObserverFactoryProvider);

    /**
     * Builds the game.
     * 
     * @param id the ID of the game
     * @return the game
     * @throws GameException if the game cannot be built
     * @throws InterruptedException if the game is interrupted
     */
    @Override
    IC4Game build(int id) throws GameException, InterruptedException;
}
