package de.fhdw.gaming.ipspiel23.c4.gui.impl;


import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;
import de.fhdw.gaming.ipspiel23.c4.gui.C4BoardEventProvider;
import de.fhdw.gaming.ipspiel23.c4.gui.event.C4BoardEvent;
import de.fhdw.gaming.ipspiel23.c4.gui.event.C4MakeMoveBoardEvent;
import javafx.application.Platform;
import javafx.scene.Cursor;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

/**
 * Implements user interaction with a C4 board.
 */
public class C4BoardEventProviderImpl implements C4BoardEventProvider {

    /**
     * the {@link C4BoardView}.
     */
    private final C4BoardView boardView;
    
    /**
     * Constructor.
     * @param boardView the {@link C4BoardView}.
     */
    public C4BoardEventProviderImpl(final C4BoardView boardView) {
        this.boardView = boardView;
    }

    @Override
    public C4BoardEvent waitForEvent(final IC4Player player, final IC4State state) {
        final Map<IC4Position, C4FieldView> legalFieldViews = new LinkedHashMap<>();
        for (final IC4Position position : state.getBoard().getLegalPositions()) {
            legalFieldViews.put(position, this.boardView.getFieldView(position).orElseThrow());
        }

        final AtomicReference<C4BoardEvent> event = new AtomicReference<>();
        final Runnable cleanUp;
        if (legalFieldViews.isEmpty()) {
            final List<C4FieldView> fieldViews = this.setupInactiveFields(state.getBoard());
            cleanUp = () -> this.cleanUpFields(fieldViews);
        } else {
            this.setupActiveFields(legalFieldViews, event);
            cleanUp = () -> this.cleanUpFields(legalFieldViews.values());
        }

        try {
            this.boardView.getUserInputSemaphore().acquire();
            return event.get();
        } catch (final InterruptedException e) {
            return null;
        } finally {
            Platform.runLater(cleanUp);
        }
    }

    /**
     * Sets up the fields when a move is possible (and hence the set of empty fields is not empty).
     *
     * @param emptyFieldViews The non-empty set of views for active fields.
     * @param event           The event to be set when the user selects an active field.
     */
    private void setupActiveFields(final Map<IC4Position, C4FieldView> emptyFieldViews,
            final AtomicReference<C4BoardEvent> event) {
        for (final Map.Entry<IC4Position, C4FieldView> entry : emptyFieldViews.entrySet()) {
            final IC4Position position = entry.getKey();
            final C4FieldView fieldView = entry.getValue();
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CROSSHAIR);
                fieldView.setHighlighted(true);
                fieldView.setOnMouseClicked((final MouseEvent mouseEvent) -> {
                    if (mouseEvent.getButton() == MouseButton.PRIMARY) {
                        event.set(new C4MakeMoveBoardEvent(position));
                        this.boardView.getUserInputSemaphore().release();
                    }
                });
            });
        }
    }
    
    /**
     * Sets up the fields when no move is possible (and hence the set of empty fields is empty).
     *
     * @param board The C4-Board Toe board.
     * @return A list of all field views on the board.
     */
    private List<C4FieldView> setupInactiveFields(final IC4Board board) {
        final List<IC4Field> fields = new ArrayList<>();
        for (final IC4Field[] ic4Fields : board.getFields()) {
            fields.addAll(Arrays.asList(ic4Fields));
        }

        final List<C4FieldView> fieldViews = new ArrayList<>();
        for (final IC4Field field : fields) {
            final C4FieldView fieldView = this.boardView.getFieldView(field.getBoardPosition()).orElseThrow();
            fieldViews.add(fieldView);
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CLOSED_HAND);
                fieldView.setOnMouseClicked(null);
            });
        }

        return fieldViews;
    }
    /**
     * Cleans up after user interaction by resetting mouse cursors and removing event handlers.
     *
     * @param fieldViews The modified field views.
     */
    private void cleanUpFields(final Collection<? extends C4FieldView> fieldViews) {
        for (final C4FieldView fieldView : fieldViews) {
            fieldView.setCursor(Cursor.DEFAULT);
            fieldView.setHighlighted(false);
            fieldView.setOnMouseClicked(null);
        }
    }

    @Override
    public void cancelWaiting() {
        this.boardView.getUserInputSemaphore().release();
    }
}
