package de.fhdw.gaming.ipspiel23.c4.gui.impl;

import java.util.Optional;

import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.layout.Region;
import javafx.scene.paint.Color;
import javafx.util.Duration;

/**
 * Displays a C4-Field.
 */
public class C4FieldView extends Region {
    
    /**
     * field occupier or empty.
     */
    private Optional<IC4Player> occupier;
    /**
     * The canvas. 
     */
    private final Canvas canvas;
    /**
     *  The duration of the move.
     */
    private final ObjectProperty<Duration> moveDuration;
    /**
     * {@code true} if the field is highlighted, else {@code false}.
     */
    private boolean highlighted;
    
    /**
     *  Creates {@link C4FieldView}.
     *  @param occupier occupying player.
     */
    public C4FieldView(final Optional<IC4Player> occupier) {
        this.occupier = occupier;
        this.moveDuration = new SimpleObjectProperty<>(Duration.millis(500.0));
        this.highlighted = false;
        
        this.canvas = new Canvas() {

            @Override
            public boolean isResizable() {
                return true;
            }

            @Override
            public double prefWidth(final double height) {
                return 0.0;
            }

            @Override
            public double prefHeight(final double width) {
                return 0.0;
            }

            @Override
            public double maxWidth(final double height) {
                return Double.POSITIVE_INFINITY;
            }

            @Override
            public double maxHeight(final double width) {
                return Double.POSITIVE_INFINITY;
            }
        };
        
        this.canvas.widthProperty().bind(this.widthProperty());
        this.canvas.heightProperty().bind(this.heightProperty());
        this.getChildren().add(this.canvas);

        this.widthProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.draw();
                });
        this.heightProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.draw();
                });

        this.setMinSize(50.0, 50.0);
        this.setMaxSize(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
    }

    /**
     * Returns the move duration property.
     */
    ObjectProperty<Duration> moveDurationProperty() {
        return this.moveDuration;
    }
    
    
    /**
     * Sets occupier.
     * @param occupier
     */
    public void setOccupier(final Optional<IC4Player> occupier) {
        this.occupier = occupier;
        draw();
    }

    /**
     * sets if highlighted.
     * @param highlighted true/false.
     */
    public void setHighlighted(final boolean highlighted) {
        this.highlighted = highlighted;
        draw();
    }

    /**
     * draws the field.
     */
    private void draw() {

        final double size = this.getWidth();

        final GraphicsContext gc = this.canvas.getGraphicsContext2D();
        gc.setFill(this.highlighted ? Color.LIGHTSLATEGRAY : Color.DARKGRAY);
        gc.fillRect(0.0, 0.0, size, size);

        gc.setFill(Color.BLACK);
        gc.beginPath();
        gc.moveTo(0.0, 0.0);
        gc.lineTo(size, 0.0);
        gc.lineTo(size, size);
        gc.lineTo(0.0, size);
        gc.lineTo(0.0, 0.0);
        gc.closePath();
        gc.stroke();

        final double margin = size * 0.1;

        if (this.occupier.isPresent()) {
            switch (this.occupier.get().getToken()) {
            case 1:
                gc.setFill(Color.RED);
                gc.fillOval(margin, margin, size - 2 * margin, size - 2 * margin);
                break;
            case 2:
                gc.setFill(Color.YELLOW);
                gc.fillOval(margin, margin, size - 2 * margin, size - 2 * margin);
                break;
            default:
                throw new UnsupportedOperationException(
                        String.format("Unknown playerToken"));
            }
        } else {
            gc.setFill(Color.WHITE);
            gc.fillOval(margin, margin, size - 2 * margin, size - 2 * margin);
        }
    }
}
