package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.collections.IReadOnlyDictionary;
import de.fhdw.gaming.ipspiel23.c4.collections.ReadOnlyDictionary;
import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Solution;

/**
 * A naive implementation of {@link IC4Board}.
 * <p>
 * NOTE: this is a "naive" proof of concept implementation to act 
 * as a baseline for our benchmarkings. This class should never be used.
 * </p>
 */
// this is a "naive" proof of concept implementation to act
// as a baseline for our benchmarkings. This class should never be used.
// THerefore we don't care about complexity :P
// This class is legacy code will never undergo further development!
@SuppressWarnings({"PMD.GodClass", "PMD.CyclomaticComplexity", "PMD.CognitiveComplexity"})
public class C4BoardHeavy implements IC4Board {

    /**
     * The players of the game.
     */
    private final IReadOnlyDictionary<Integer, IC4Player> players;

    /**
     * The number of rows in the board.
     */
    private final int rowCount;

    /**
     * The number of columns in the board.
     */
    private final int columnCount;

    /**
     * The number of fields that must be connected to win the game.
     */
    private final int solutionSize;

    /**
     * The fields of the board.
     */
    private final IC4Field[][] fields;

    /**
     * Creates a new board.
     * 
     * @param players The players of the game.
     * @param rowCount The number of rows in the board.
     * @param columnCount The number of columns in the board.
     * @param solutionSize The number of fields that must be connected to win the game.
     */
    public C4BoardHeavy(final IReadOnlyDictionary<Integer, IC4Player> players, final int rowCount, 
        final int columnCount, final int solutionSize) {
        this.players = players;
        this.rowCount = rowCount;
        this.columnCount = columnCount;
        this.solutionSize = solutionSize;
        this.fields = new IC4Field[rowCount][columnCount];
        for (int row = 0; row < rowCount; row++) {
            for (int column = 0; column < columnCount; column++) {
                this.fields[row][column] = new C4FieldHeavy(this, new C4Position(row, column));
            }
        }
    }

    /**
     * Creates a new board.
     * 
     * @param board The board to copy.
     */
    private C4BoardHeavy(final C4BoardHeavy board) {
        final Map<Integer, IC4Player> playerMap = new HashMap<>();
        for (final Integer key : board.players.getKeys()) {
            playerMap.put(key, board.players.getValueOrDefault(key));
        }
        this.players = ReadOnlyDictionary.of(playerMap);
        this.rowCount = board.rowCount;
        this.columnCount = board.columnCount;
        this.solutionSize = board.solutionSize;
        this.fields = new IC4Field[rowCount][columnCount];
        for (int row = 0; row < rowCount; row++) {
            for (int column = 0; column < columnCount; column++) {
                this.fields[row][column] = board.getField(new C4Position(row, column)).deepCopy();
            }
        }
    }

    @Override
    public int getRowCount() {
        return this.rowCount;
    }

    @Override
    public int getColumnCount() {
        return this.columnCount;
    }

    @Override
    public int getMinimumSolutionSize() {
        return this.solutionSize;
    }

    @Override
    public boolean checkBounds(final int row, final int column) {
        return row >= 0 && row < this.rowCount && column >= 0 && column < this.columnCount;
    }

    @Override
    public boolean isFull() {
        for (int i = 0; i < this.getRowCount(); i++) {
            for (int j = 0; j < this.getColumnCount(); j++) {
                final Optional<IC4Field> field = tryGetField(new C4Position(i, j));
                if (field.get().getOccupyingPlayer().isEmpty()) {
                    return false;
                }
            }
        }
        return true;
    }

    @Override
    public IC4BoardSlim getInternalBoard() {
        throw new UnsupportedOperationException("'getInternalBoard' is not supported by this implementation");
    }

    @Override
    public Optional<IC4Field> tryGetField(final IC4Position position) {
        if (checkBounds(position)) {
            return Optional.of(this.fields[position.getRow()][position.getColumn()]);
        }
        return Optional.empty();
    }

    @Override
    public IC4Field getField(final IC4Position position) {
        if (checkBounds(position)) {
            return tryGetField(position).get();
        }
        throw new IndexOutOfBoundsException("Position is out of bounds");
    }

    @Override
    public IC4Field[][] getFields() {
        final IC4Field[][] copiedFields = new IC4Field[this.fields.length][];
        for (int i = 0; i < this.fields.length; i++) {
            copiedFields[i] = this.fields[i].clone();
        }
        return copiedFields;
    }

    @Override
    public Optional<IC4Solution> tryFindFirstSolution() {
        final Set<IC4Solution> solutions = findAllSolutionsInternal(false);
        if (solutions.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(solutions.iterator().next());
    }

    @Override
    public Set<IC4Solution> findAllSolutions() {
        return findAllSolutionsInternal(true);
    }
    
    /**
     * Scans all directions for solutions.
     * 
     * @param eagerEvaluation Whether to stop scanning after the first solution was found.
     * @return The set of solutions to add the found solutions to.
     */
    private Set<IC4Solution> findAllSolutionsInternal(final boolean eagerEvaluation) {
        final Set<IC4Solution> solutions = new HashSet<>();
        
        scanHorizontal(solutions, eagerEvaluation);
        scanVertical(solutions, eagerEvaluation);
        scanDiagonalLeft(solutions, eagerEvaluation);
        scanDiagonalRight(solutions, eagerEvaluation);

        return solutions;
    }

    /**
     * Scans all horizontal lanes for solutions.
     * 
     * @param solutions The set of solutions to add the found solutions to.
     * @param eagerEvaluation Whether to stop scanning after the first solution was found.
     */
    private void scanHorizontal(final Set<IC4Solution> solutions, final boolean eagerEvaluation) {
        if (!eagerEvaluation && !solutions.isEmpty()) {
            return;
        }
        boolean foundSolution = false;
        for (int row = 0; row < this.getRowCount() && (!foundSolution || eagerEvaluation); row++) {
            final IC4Position startPosition = new C4Position(row, 0);
            foundSolution = scanLane(solutions, startPosition, C4Direction.EAST, eagerEvaluation);
        }
    }

    /**
     * Scans all vertical lanes for solutions.
     * 
     * @param solutions The set of solutions to add the found solutions to.
     * @param eagerEvaluation Whether to stop scanning after the first solution was found.
     */
    private void scanVertical(final Set<IC4Solution> solutions, final boolean eagerEvaluation) {
        if (!eagerEvaluation && !solutions.isEmpty()) {
            return;
        }
        boolean foundSolution = false;
        for (int column = 0; column < this.getColumnCount() && (!foundSolution || eagerEvaluation); column++) {
            final IC4Position startPosition = new C4Position(0, column);
            foundSolution = scanLane(solutions, startPosition, C4Direction.SOUTH, eagerEvaluation);
        }
    }

    /**
     * Scans all diagonals from top left top bottom right for solutions.
     * 
     * @param solutions The set of solutions to add the found solutions to.
     * @param eagerEvaluation Whether to stop scanning after the first solution was found.
     */
    private void scanDiagonalRight(final Set<IC4Solution> solutions, final boolean eagerEvaluation) {
        if (!eagerEvaluation && !solutions.isEmpty()) {
            return;
        }
        boolean foundSolution = false;
        for (int row = 0; row < this.getRowCount() && (!foundSolution || eagerEvaluation); row++) {
            final IC4Position startPosition = new C4Position(row, 0);
            foundSolution = scanLane(solutions, startPosition, C4Direction.SOUTH_EAST, eagerEvaluation);
        }
        for (int column = 1; column < this.getColumnCount() && (!foundSolution || eagerEvaluation); column++) {
            final IC4Position startPosition = new C4Position(0, column);
            foundSolution = scanLane(solutions, startPosition, C4Direction.SOUTH_EAST, eagerEvaluation);
        }
    }

    /**
     * Scans all diagonals from top right to bottom left for solutions.
     * 
     * @param solutions The set of solutions to add the found solutions to.
     * @param eagerEvaluation Whether to stop scanning after the first solution was found.
     */
    private void scanDiagonalLeft(final Set<IC4Solution> solutions, final boolean eagerEvaluation) {
        if (!eagerEvaluation && !solutions.isEmpty()) {
            return;
        }
        boolean foundSolution = false;
        for (int row = 0; row < getRowCount() && (!foundSolution || eagerEvaluation); row++) {
            final IC4Position startPosition = new C4Position(row, this.getColumnCount() - 1);
            foundSolution = scanLane(solutions, startPosition, C4Direction.SOUTH_WEST, eagerEvaluation);
        }
        for (int column = 0; column < this.getColumnCount() - 1 && (!foundSolution || eagerEvaluation); column++) {
            final IC4Position startPosition = new C4Position(0, column);
            foundSolution = scanLane(solutions, startPosition, C4Direction.SOUTH_WEST, eagerEvaluation);
        }
    }

    /**
     * Scans a lane for solutions.
     * 
     * @param solutions The set of solutions to add the found solutions to.
     * @param startPosition The position to start scanning from.
     * @param direction The direction to scan in.
     * @param eagerEvaluation Whether to stop scanning after the first solution was found.
     * @return Whether a solution was found.
     */
    private boolean scanLane(final Set<IC4Solution> solutions, final IC4Position startPosition, 
            final C4Direction direction, final boolean eagerEvaluation) {
        if (!eagerEvaluation && !solutions.isEmpty()) {
            return true;
        }
        
        Optional<IC4Player> lastPlayer = Optional.empty();
        int lastPlayerCount = 0;
        
        IC4Position currentPosition = startPosition;
        while (true) {
            final Optional<IC4Field> field = this.tryGetField(currentPosition);
            if (field.isEmpty()) {
                break;
            }
            final Optional<IC4Player> currentPlayer = field.get().getOccupyingPlayer();
            if (currentPlayer.isPresent()) {
                if (currentPlayer.equals(lastPlayer)) {
                    lastPlayerCount++;
                } else {
                    lastPlayer = currentPlayer;
                    lastPlayerCount = 1;
                }
                if (lastPlayerCount >= this.solutionSize) {
                    final var solution = scanFullSolution(lastPlayerCount, lastPlayer.get(), currentPosition, 
                        direction);
                    solutions.add(solution);
                    if (!eagerEvaluation) {
                        return true;
                    }
                }
            } else {
                lastPlayer = Optional.empty();
                lastPlayerCount = 0;
            }

            currentPosition = direction.stepFrom(currentPosition, 1);
        }
        return false;
    }

    /**
     * Scans a full solution to include as many fields as possible.
     * 
     * @param currentCount The current count of fields in the solution.
     * @param player The player that owns the solution.
     * @param currentPosition The current position.
     * @param direction The direction to scan in.
     * @return The solution.
     */
    private IC4Solution scanFullSolution(final int currentCount, final IC4Player player, 
            final IC4Position currentPosition, final C4Direction direction) {
        final int size = currentCount + scanRemaining(currentPosition, direction, player);
        final IC4Field[] solutionFields = new IC4Field[size];
        final IC4Position startPosition = direction.getInverse().stepFrom(currentPosition, currentCount - 1);
        for (int i = 0; i < size; i++) {
            final Optional<IC4Field> field = tryGetField(direction.stepFrom(startPosition, i));
            solutionFields[i] = field.get();
        }
        return new C4SolutionHeavy(player, startPosition, direction.stepFrom(startPosition, size - 1), 
            direction, solutionFields);
    }

    /**
     * Scans the remaining fields in a direction.
     * 
     * @param position The position to start scanning from.
     * @param direction The direction to scan in.
     * @param player The player that owns the solution.
     * @return The number of fields that can be added to the solution.
     */
    private int scanRemaining(final IC4Position position, final C4Direction direction, final IC4Player player) {
        int count = 0;
        IC4Position currentPosition = position;
        while (true) {
            currentPosition = direction.stepFrom(currentPosition, 1);
            final Optional<IC4Field> currentField = tryGetField(currentPosition);
            if (currentField.isEmpty() || currentField.get().getOccupyingPlayer().isEmpty()) {
                break;
            }
            final Optional<IC4Player> fieldPlayer = currentField.get().getOccupyingPlayer();
            if (fieldPlayer.isEmpty()) {
                break;
            }
            if (fieldPlayer.get().equals(player)) {
                count++;
            } else {
                break;
            }
        }
        return count;
    }

    @Override
    public boolean isEmpty(final IC4Position position) {
        return this.fields[position.getRow()][position.getColumn()].getOccupyingPlayer().isEmpty();
    }

    @Override
    public boolean isEmpty(final int row, final int column) {
        return isEmpty(new C4Position(row, column));
    }

    @Override
    public boolean checkBounds(final IC4Position position) {
        return checkBounds(position.getRow(), position.getColumn());
    }

    @Override
    public boolean isSolid(final IC4Position position) {
        // we need to also shift the row by one to the top because 
        // the row below the board is also considered solid and
        // therefore for the context of this method, the board is
        // one row higher than it actually is
        if (!this.checkBounds(position) && !this.checkBounds(C4Direction.NORTH.stepFrom(position, 1))) {
            throw new IndexOutOfBoundsException("The provided position is not within the defined bounds of the board!");
        }
        final Optional<IC4Field> field = tryGetField(position);
        if (field.isPresent()) {
            return field.get().getOccupyingPlayer().isPresent();
        }
        // bottom border is always solid
        return tryGetField(C4Direction.NORTH.stepFrom(position, 1)).isPresent();
    }

    @Override
    public boolean isSolid(final int row, final int column) {
        return isSolid(new C4Position(row, column));
    }

    @Override
    public Optional<IC4Player> getOccupyingPlayerOrDefault(final IC4Position position) {
        if (!this.checkBounds(position)) {
            throw new IndexOutOfBoundsException("The provided position is not within the defined bounds of the board!");
        }
        final Optional<IC4Field> field = tryGetField(position);
        if (field.isPresent()) {
            return field.get().getOccupyingPlayer();
        }
        return Optional.empty();
    }

    @Override
    public IC4Position[] getEmptyPositions() {
        final List<IC4Position> emptyPositions = new ArrayList<>();
        for (int row = 0; row < getRowCount(); row++) {
            for (int column = 0; column < getColumnCount(); column++) {
                final IC4Position position = new C4Position(row, column);
                if (isEmpty(position)) {
                    emptyPositions.add(position);
                }
            }
        }
        return emptyPositions.toArray(new IC4Position[0]);
    }

    @Override
    public IC4Position[] getPositionsByPlayer(final IC4Player player) {
        final List<IC4Position> positions = new ArrayList<>();
        for (int row = 0; row < getRowCount(); row++) {
            for (int column = 0; column < getColumnCount(); column++) {
                final IC4Position position = new C4Position(row, column);
                final Optional<IC4Player> occupyingPlayer = getOccupyingPlayerOrDefault(position);
                if (occupyingPlayer.isPresent() && occupyingPlayer.get().equals(player)) {
                    positions.add(position);
                }
            }
        }
        return positions.toArray(new IC4Position[0]);
    }

    @Override
    public IC4Position[] getLegalPositions() {
        final List<IC4Position> legalPositions = new ArrayList<>();
        boolean inAir = false;
        for (int column = 0; column < getColumnCount(); column++) {
            int row = this.getRowCount() - 1;
            for (; row >= 0 && !inAir; row--) {
                inAir = this.getField(new C4Position(row, column)).getOccupyingPlayer().isEmpty();
            }
            if (inAir) {
                inAir = false;
                legalPositions.add(new C4Position(row + 1, column));
            }
        }
        return legalPositions.toArray(new IC4Position[0]);
    }

    @Override
    public int countEmptyPositions() {
        return getEmptyPositions().length;
    }

    @Override
    public int countPositionsByPlayer(final IC4Player player) {
        return getPositionsByPlayer(player).length;
    }

    @Override
    public int countLegalPositions() {
        return getLegalPositions().length;
    }

    @Override
    public IC4Board deepCopy() {
        return new C4BoardHeavy(this);
    }
}