package de.fhdw.gaming.ipspiel23.c4.domain.impl.evaluation;

import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4SolutionSlim;

import static de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim.EMPTY_TOKEN;

/**
 * A {@link C4SolutionAnalyzer} that analyzes the board horizontally (rows).
 */
public class C4SolutionAnalyzerHorizontal extends C4SolutionAnalyzer {

    /**
     * The last row which is updated while searching for a solution to 
     * speed up the search process.
     */
    private int optimizedRowMax;

    /**
     * Creates a new {@link C4SolutionAnalyzerHorizontal}.
     * @param board The board to analyze.
     */
    public C4SolutionAnalyzerHorizontal(final C4BoardSlim board) {
        super(board, C4Direction.EAST);
        optimizedRowMax = super.rowMax();
    }

    @Override
    public IC4SolutionSlim tryFindFirstSolution(final IC4SolutionSlim currentSolution, final boolean updateCache) {

        if (currentSolution != null) {
            return currentSolution;
        }
        
        boolean isRowEmpty = false;
        // count row by row, EAST from bottom to top for faster matches
        // abort as soon as we reach an empty row (gravity)
        for (int row = this.optimizedRowMax - 1; row >= 0 && !isRowEmpty; row--) {
            int token;
            int lastToken = 0;
            int count = 0;
            isRowEmpty = true;
            boolean isRowFull = true;
            for (int col = 0; col < colMax(); col++, lastToken = token) {
                token = board().getTokenUnsafe(row, col);
                count = countConsecutivesBranchless(count, token, lastToken);
                if (count >= targetCount()) {
                    return scanRemaining(token, row, col);
                }
                isRowFull &= token != EMPTY_TOKEN;
                isRowEmpty &= token == EMPTY_TOKEN;
            }
            // if the current row is full and contains no solution we can remember that
            // and don't have to re-evaluate that row next time
            if (updateCache && isRowFull) {
                this.optimizedRowMax = row;
            }
        }
        return null;
    }

    @Override
    public void findAllSolutions(final Set<IC4SolutionSlim> resultSet, final boolean updateCache) {
        // count row by row, EAST from bottom to top for faster matches
        boolean isRowEmpty = false;
        for (int row = this.optimizedRowMax - 1; row >= 0 && !isRowEmpty; row--) {
            isRowEmpty = true;
            int token;
            int lastToken = 0;
            int count = 0;
            boolean isRowFull = true;
            boolean rowContainsSolution = false;
            int col = 0;
            for (; col < colMax(); col++, lastToken = token) {
                token = board().getTokenUnsafe(row, col);
                count = countConsecutivesBranchless(count, token, lastToken);
                if (count >= targetCount()) {
                    count = 0;
                    final C4SolutionSlim solution = scanRemaining(token, row, col);
                    resultSet.add(solution);
                    rowContainsSolution = true;
                    // skip to the end of the solution
                    col = solution.getColumnIndexEnd();
                }
                isRowFull &= token != EMPTY_TOKEN;
                isRowEmpty &= token == EMPTY_TOKEN;
            }
            // if the current row is full and contains no solution we can remember that
            // and don't have to re-evaluate that row the next time
            if (updateCache && isRowFull && !rowContainsSolution) {
                this.optimizedRowMax = row;
            }
        }
    }
    
    @Override
    protected C4SolutionSlim scanRemaining(final int token, final int startRow, final int startCol) {
        int col = startCol + 1;
        while (col < colMax() && board().getTokenUnsafe(startRow, col) == token) {
            col++;
        }
        // revert last change to get inclusive upper bound
        col--;
        return solutionOf(token, startRow, col, board().getMinimumSolutionSize() + col - startCol);
    }

    @Override
    public void resetCache() {
        this.optimizedRowMax = super.rowMax();
    }
}
