package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel23.c4.collections.IReadOnlyDictionary;
import de.fhdw.gaming.ipspiel23.c4.collections.ReadOnlyDictionary;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Game;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4GameBuilder;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4PlayerBuilder;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;
import de.fhdw.gaming.ipspiel23.c4.moves.IC4Move;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;

/**
 * The {@link IC4GameBuilder} implementation.
 */
public final class C4GameBuilder implements IC4GameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;

    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * The number of rows of the board.
     */
    private int rowCount;

    /**
     * The number of columns of the board.
     */
    private int columnCount;

    /**
     * The required solution size.
     */
    private int solutionSize;

    /**
     * The current player id used to identify the next player.
     */
    private final AtomicInteger currentPlayerId;

    /**
     * The players added to the game so far.
     */
    private final List<IC4Player> players;

    /**
     * The player builders used to create the players.
     */
    private final Map<Integer, C4PlayerBuilder> playerBuilders;

    /**
     * Creates a new {@link C4GameBuilder} instance.
     */
    public C4GameBuilder() {
        this.players = new ArrayList<>(2);
        this.currentPlayerId = new AtomicInteger(0);
        this.playerBuilders = new HashMap<>(2);
        this.maxComputationTimePerMove = IC4GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
        this.rowCount = IC4GameBuilder.DEFAULT_BOARD_ROWS;
        this.columnCount = IC4GameBuilder.DEFAULT_BOARD_COLUMNS;
        this.solutionSize = IC4GameBuilder.DEFAULT_REQUIRED_SOLUTION_SIZE;
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
    }

    @Override
    public C4GameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        if (newMaxComputationTimePerMove <= 0) {
            throw new IllegalArgumentException("The computation time must be positive.");
        }
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public IC4PlayerBuilder createPlayerBuilder() {
        final int playerId = currentPlayerId.incrementAndGet();
        final C4PlayerBuilder builder = new C4PlayerBuilder(playerId);
        // store player builder to be able to inject the strategy into the player object
        // after instantiation
        this.playerBuilders.put(playerId, builder);
        return builder;
    }

    @Override
    public IC4GameBuilder addPlayer(final IC4Player player, final IC4Strategy strategy) throws GameException {
        Objects.requireNonNull(player, "player");
        Objects.requireNonNull(strategy, "strategy");
        final C4PlayerBuilder builder = this.playerBuilders.get(player.getToken());
        if (builder == null) {
            throw new GameException(String.format("Encountered player %s with invalid or expired token: %s", 
                player, player.getToken()));
        }
        for (final var p : this.players) {
            if (player.getName().equalsIgnoreCase(p.getName())) {
                throw new GameException(String.format("Tried to add two players with identical name: %s", 
                    player.getName()));
            }
        }
        // we need a way to store the strategy in the player instance (because that's where it belongs)
        // that is handled via an internal callback set in the ctor of the player.
        // We don't want client code to be able to modify strategies, so no simple public setters :)
        builder.injectPlayerStrategyUsingHook(strategy);
        this.playerBuilders.remove(player.getToken());
        this.players.add(player);
        return this;
    }

    @Override
    public IC4GameBuilder changeBoardRows(final int newRowCount) {
        this.rowCount = newRowCount;
        return this;
    }

    @Override
    public IC4GameBuilder changeBoardColumns(final int newColumnCount) {
        this.columnCount = newColumnCount;
        return this;
    }

    @Override
    public IC4GameBuilder changeRequiredSolutionSize(final int newSolutionSize) {
        this.solutionSize = newSolutionSize;
        return this;
    }

    @Override
    public IC4GameBuilder changeObserverFactoryProvider(final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public IC4Game build(final int id) throws GameException, InterruptedException {
        // it's questionable how exciting a Connect Four game with a single player would be, but its fine by me
        // who am I to judge ¯\_(ツ)_/¯
        if (this.players.isEmpty()) {
            throw new GameException("A Connect Four game needs players to..., you know..., qualify as a game :P");
        }
        // ensure players are ordered.
        Collections.sort(this.players, (p1, p2) -> p1.getToken() - p2.getToken());
        // now assert player IDs are consecutive, unique, and start at 1
        // we do the same check in the C4State ctor, but better be safe than sorry
        // at the same time:
        //  - create the token -> player Look Up Table (LUT) for the internal board
        //  - create the name -> strategy LUT for the gaming framework
        final IC4Player[] playerArr = new IC4Player[this.players.size()];
        this.players.toArray(playerArr);
        final Map<Integer, IC4Player> playerTokenLut = new HashMap<>(playerArr.length);
        final Map<String, IC4Strategy> strategyLut = new HashMap<>(playerArr.length);
        for (int i = 0; i < playerArr.length; i++) {
            final IC4Player player = playerArr[i];
            if (player.getToken() != i + 1) {
                throw new IllegalStateException(String.format("The 'players' must be ordered ascending by tokens, "
                    + "starting at token 1 and incrementing by 1 for each following player!. Player %s at index %s "
                    + "violates this rule!", 
                    player, i));
            }
            playerTokenLut.put(player.getToken(), player);
            strategyLut.put(player.getName(), player.getStrategy());
        }
        final IReadOnlyDictionary<Integer, IC4Player> roPlayerTokenLut = ReadOnlyDictionary.of(playerTokenLut);
        final IC4BoardSlim slimBoard = new C4BoardSlim(roPlayerTokenLut, this.rowCount, this.columnCount, 
            this.solutionSize);
        final IC4Board board = new C4Board(slimBoard);
        final IC4State initialState = new C4State(playerArr, board);
        return new C4Game(id, initialState, strategyLut, this.maxComputationTimePerMove,  
            // some black magic going on here, but it's fine, or so I'm told
            // no idea how Java manages to type cast a boolean member of a completely
            // unrelated class to an IMoveChecker instance, but apparently it does
               // ¯\_(ツ)_/¯
            // (why am I even using Java again?)
            // feel free to enlighten me if you know what's going on here :P
            IC4Move.class::isInstance,
            observerFactoryProvider);
    }
}
