package de.fhdw.gaming.ipspiel23.c4.domain.impl.evaluation;

import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4SolutionSlim;

import static de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim.EMPTY_TOKEN;

/**
 * A {@link C4SolutionAnalyzer} that analyzes the board vertically (columns).
 */
public class C4SolutionAnalyzerVertical extends C4SolutionAnalyzer {

    /**
     * A blacklist containing all column indexes which are not part of a solution.
     */
    private final boolean[] isColumnBlacklisted;

    /**
     * Creates a new {@link C4SolutionAnalyzerVertical}.
     * @param board The board to analyze.
     */
    public C4SolutionAnalyzerVertical(final C4BoardSlim board) {
        super(board, C4Direction.NORTH);
        isColumnBlacklisted = new boolean[colMax()];
    }

    @Override
    public void resetCache() {
        for (int i = 0; i < isColumnBlacklisted.length; i++) {
            isColumnBlacklisted[i] = false;
        }
    }

    @Override
    public IC4SolutionSlim tryFindFirstSolution(final IC4SolutionSlim currentSolution, final boolean updateCache) {
        
        if (currentSolution != null) {
            return currentSolution;
        }
        
        // search columns (bottom to top)
        // count column by column, bottom to top from left to right for faster matches
        for (int col = 0; col < colMax(); col++) {
            // skip columns that are full and of which we know that they don't contain a solution.
            if (isColumnBlacklisted[col]) {
                continue;
            }
            // continue with next column when we start finding empty fields (gravity)
            boolean inAir = false;
            int count = 0;
            int lastToken = 0;

            for (int row = rowMax() - 1; row >= 0 && !inAir; row--) {
                final int token = board().getTokenUnsafe(row, col);
                count = countConsecutivesBranchless(count, token, lastToken);
                if (count >= targetCount()) {
                    return scanRemaining(token, row, col);
                }
                lastToken = token;
                inAir = token == EMPTY_TOKEN;
            }
            // add column to blacklist iff we broke out of the loop 
            // because the row index was < 0 (column full)
            // and we're allowed to update the cache
            isColumnBlacklisted[col] = updateCache && !inAir;
        }
        return null;
    }

    @Override
    public void findAllSolutions(final Set<IC4SolutionSlim> resultSet, final boolean updateCache) {
        // search columns (bottom to top)
        // count column by column, bottom to top from left to right for faster matches
        for (int col = 0; col < colMax(); col++) {
            // skip columns that are full and of which we know that they don't contain a solution.
            if (isColumnBlacklisted[col]) {
                continue;
            }
            // continue with next column when we start finding empty fields (gravity)
            boolean inAir = false;
            int count = 0;
            int lastToken = 0;
            boolean colContainsSolution = false;
            int row = rowMax() - 1;
            for (; row >= 0 && !inAir; row--) {
                final int token = board().getTokenUnsafe(row, col);
                count = countConsecutivesBranchless(count, token, lastToken);
                if (count >= targetCount()) {
                    final C4SolutionSlim solution = scanRemaining(token, row, col);
                    resultSet.add(solution);
                    count = 0;
                    colContainsSolution = true;
                    // skip to end of the match
                    row = solution.getRowIndexEnd();
                }
                lastToken = token;
                inAir = token == 0;
            }
            // this column is blacklisted iff 
            // 1. We are allowed to update the cache
            // 2. we never reached an empty field (!inAir == column is full)
            // 3. this column doesn't contain a solution
            isColumnBlacklisted[col] = updateCache && !inAir && !colContainsSolution;
        }
    }
    
    @Override
    protected C4SolutionSlim scanRemaining(final int token, final int startRow, final int startCol) {
        int row = startRow - 1;
        while (row >= 0 && board().getTokenUnsafe(row, startCol) == token) {
            row--;
        }
        // revert last change to get inclusive upper bound
        row++;
        return solutionOf(token, row, startCol, board().getMinimumSolutionSize() + startRow - row);
    }
}
