package de.fhdw.gaming.ipspiel23.c4.domain;

import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4Position;

/**
 * This class provides methods to rematerialize and dematerialize {@link IC4Position} instances.
 * <p>
 * Note: We apply the principle of {@link https://en.wikipedia.org/wiki/Rematerialization} to
 * memory management. Only instead of worrying about spilling registers to memory, we worry about
 * spilling row and column stack values to the heap.
 * </p>
 * <p>
 * This means that we dematerialize position objects to primitive long values for efficient storage 
 * in arrays and rematerialize them back into position objects when needed. This is done to prevent 
 * unnecessary heap allocations for position instances, reducing memory overhead and garbage collection 
 * pressure. Dematerialized positions allow for inlining of position instances in arrays for better 
 * cache locality and performance as they are value types.
 * </p>
 */
// C# has structs, Java does not. Therefore, we have this abomination.
public final class C4PositionMaterializer {

    /**
     * The unused private ctor of this "static" class.
     */
    private C4PositionMaterializer() { }
    
    /**
     * Rematerializes a dematerialized position from a primitive long value back into a position
     * object.
     * @param dematPosition The dematerialized position.
     * @return The rematerialized position.
     */
    public static IC4Position rematerialize(final long dematPosition) {
        return new C4Position((int) (dematPosition >> 32), (int) dematPosition);
    }

    /**
     * Dematerializes a position object into a primitive long value.
     * @param position The position to dematerialize.
     * @return The dematerialized position.
     */
    public static long dematerialize(final IC4Position position) {
        return dematerialize(position.getRow(), position.getColumn());
    }

    /**
     * Dematerializes a position row and column index into a primitive long value.
     * @param row The row of the position.
     * @param column The column of the position.
     * @return The dematerialized position.
     */
    public static long dematerialize(final int row, final int column) {
        return ((long) row) << 32 | column;
    }

    /**
     * Gets the row.
     * @param pos the position as a long.
     * @return The row.
     */
    public static int getRow(final long pos) {
        return (int) (pos >> 32);
    }

    /**
     * Gets the column.
     * @param pos the position as a long.
     * @return The column.
     */
    public static int getColumn(final long pos) {
        return (int) pos;
    }
}
