package de.fhdw.gaming.ipspiel23.c4.domain;

import java.util.Set;


/**
 * The lightweight internal representation of the connect four board, somewhat optimized for performance.
 */
public interface IC4BoardSlim extends IC4BoardBase {
    /**
    * A lightweight representation of the board-state in a single contiguous region of memory (data locality) 
    * for fast memory access and better data locality. Has getRowCount() rows and getColumnCount() columns.
    * This method exposes the internal board state and should be used with care.
    */
    int[] getBoardStateUnsafe();

    /**
     * Retrieves the token used to represent an empty field.
     */
    int emptyToken();

    /**
     * Retrieves the token at the specified position without checking the bounds.
     * 
     * @param row the row of the position
     * @param column the column of the position
     * @return the token at the specified position
     * @throws ArrayIndexOutOfBoundsException if {@code row * getColumnCount() + column} is out of bounds; 
     * other row or column violations are not checked
     */
    int getTokenUnsafe(int row, int column);

    /**
     * Updates the token at the specified position without checking the bounds.
     * 
     * @param row the row index of the position
     * @param column the column index of the position
     * @param token the token to write to the specified position
     * @throws ArrayIndexOutOfBoundsException if {@code position.getRow() * getColumnCount() + position.getColumn()} 
     * is out of bounds; 
     * other row or column violations are not checked
     */
    void updateTokenUnsafe(int row, int column, int token);

    /**
     * Retrieves the player uniquely identified by the specified token.
     * @param token the token
     * @return the player, or null if no player is associated with the token
     */
    IC4Player getPlayerByToken(int token);
    
    /**
     * Resets all caches of the solution analyzers causing lines determined being free of any
     * solutions to be re-analyzed the next time a solution is searched.
     */
    void resetSolutionAnalyzerCaches();

    /**
     * Lazy evaluation for the first solution, e.g. the first n-in-a-row (where n is the number of 
     * tokens in a row required to win)
     * @return the first solution, or null if no solution exists for the current board state
     */
    IC4SolutionSlim tryFindFirstSolution();

    /**
     * Lazy evaluation for the first solution, e.g. the first n-in-a-row (where n is the number of 
     * tokens in a row required to win)
     * @param updateCache Whether to update the solution cache, preventing lines determined to not 
     * contain any solutions from being scanned again.
     * @return the first solution, or null if no solution exists for the current board state
     */
    IC4SolutionSlim tryFindFirstSolution(boolean updateCache);
    
    /**
     * Eager evaluation for all solutions, e.g. all n-in-a-row (where n is the number of 
     * tokens in a row required to win)
     * @return all solutions, or an empty set if no solution exists for the current board state
     */
    Set<IC4SolutionSlim> findAllSolutions();

    /**
     * Eager evaluation for all solutions, e.g. all n-in-a-row (where n is the number of 
     * tokens in a row required to win)
     * @param updateCache Whether to update the solution cache, preventing lines determined to not 
     * contain any solutions from being scanned again.
     * @return all solutions, or an empty set if no solution exists for the current board state
     */
    Set<IC4SolutionSlim> findAllSolutions(boolean updateCache);

    /**
     * Checks whether the specified position is empty without checking the bounds.
     * 
     * @param row the row of the position
     * @param column the column of the position
     * @return true if the specified position is empty, false otherwise
     * @throws ArrayIndexOutOfBoundsException if {@code row * getColumnCount() + column} is out of bounds; 
     * other row or column violations are not checked
     */
    boolean isEmptyUnsafe(int row, int column);

    /**
     * Checks whether the specified position is "solid" without checking the bounds. 
     * A position is solid if it is not empty, or if it is one row below the bottom row (i.e., it is the "floor")
     * 
     * @param row the row of the position
     * @param column the column of the position
     * @return true if the specified position is solid, false if it is empty ("air") or out of bounds
     */
    boolean isSolidUnsafe(int row, int column);

    @Override
    IC4BoardSlim deepCopy();

    /**
     * Writes the positions of all tokens matching the specified token to the specified buffer.
     * Positions are written in a "dematerialized" format, i.e. the row and column of each position are written
     * as a single long value, where the row is stored in the upper 32 bits and the column in the lower 32 bits.
     * This format is used to improve performance by not having to allocate a new object for each position.
     * Also complies with the data locality principle, i.e. all positions are written to a single contiguous region 
     * of memory for faster memory access.
     * @param buffer a reference to the buffer to write the positions to
     * @param token the token to match
     * @return the number of positions written to the buffer
     * @see C4PositionMaterializer
     * @throws ArrayIndexOutOfBoundsException if the buffer is too small to hold all positions.
     */
    int getDematPositionsByTokenUnsafe(long[] buffer, int token);

    /**
     * Writes the positions of all empty fields to the specified buffer.
     * Positions are written in a "dematerialized" format, i.e. the row and column of each position are written
     * as a single long value, where the row is stored in the upper 32 bits and the column in the lower 32 bits.
     * This format is used to improve performance by not having to allocate a new object for each position.
     * Also complies with the data locality principle, i.e. all positions are written to a single contiguous region 
     * of memory for faster memory access.
     * @param buffer A reference to the buffer to write the positions to
     * @return the number of positions written to the buffer
     * @see C4PositionMaterializer
     * @throws ArrayIndexOutOfBoundsException if the buffer is too small to hold all positions.
     */
    // suppress "UseVarargs: Consider using varargs for methods or constructors which take 
    // an array the last parameter."
    // using varargs here would be fatal because results are returned via the buffer...
    @SuppressWarnings("PMD.UseVarargs")
    int getLegalDematPositionsUnsafe(long[] buffer);
}
