package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.collections.IReadOnlyDictionary;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.C4PositionMaterializer;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.evaluation.C4SolutionEvaluator;

/**
 * The lightweight internal representation of the connect four board, somewhat optimized for performance.
 */
public class C4BoardSlim implements IC4BoardSlim {

    /**
     * The empty token value.
     */
    public static final int EMPTY_TOKEN = 0;

    /**
     * The number of rows in the board.
     */
    private final int rowCount;

    /**
     * The number of columns in the board.
     */
    private final int columnCount;

    /**
     * The flattened one-dimensional board state/token array.
     */
    private final int[] board;

    /**
     * A lookup table for players by their token.
     */
    private final IReadOnlyDictionary<Integer, IC4Player> players;

    /**
     * The number of tokens in a row required to win.
     */
    private final int solutionSize;

    /**
     * The solution evaluator.
     */
    private final C4SolutionEvaluator evaluator;
    
    /**
     * Creates a new board with the specified dimensions and solution size.
     * @param players the players
     * @param rowCount the number of rows
     * @param columnCount the number of columns
     * @param solutionSize the number of tokens in a row required to win
     */
    public C4BoardSlim(final IReadOnlyDictionary<Integer, IC4Player> players, final int rowCount, 
        final int columnCount, final int solutionSize) {
        this.players = players;
        this.rowCount = rowCount;
        this.columnCount = columnCount;
        this.board = new int[rowCount * columnCount];
        this.solutionSize = solutionSize;
        this.evaluator = new C4SolutionEvaluator(this);
    }
    
    /**
     * Creates a new board with the specified dimensions and solution size. Literally only used for unit testing :P
     * @param board the board state
     * @param players the players
     * @param rowCount the number of rows
     * @param columnCount the number of columns
     * @param solutionSize the number of tokens in a row required to win
     */
    // we only use this ctor for unit testing to, yes directly inject an array in this class.
    // It's just easier this way. So this is justified. Also: this ctor is not public, so all
    // is fine :)
    @SuppressWarnings("PMD.ArrayIsStoredDirectly")
    public C4BoardSlim(final int[] board, final IReadOnlyDictionary<Integer, IC4Player> players, final int rowCount, 
            final int columnCount, final int solutionSize) {
        this.board = board;
        this.players = players;
        this.rowCount = rowCount;
        this.columnCount = columnCount;
        this.solutionSize = solutionSize;
        this.evaluator = new C4SolutionEvaluator(this);
    }
    
    /**
     * Copy constructor.
     * @param original the original board
     */
    private C4BoardSlim(final C4BoardSlim original) {
        this.players = original.players;
        this.rowCount = original.rowCount;
        this.columnCount = original.columnCount;
        this.board = original.board.clone();
        this.solutionSize = original.solutionSize;
        this.evaluator = new C4SolutionEvaluator(this);
    }

    @Override
    public int emptyToken() {
        return EMPTY_TOKEN;
    }
    
    @Override
    public int getRowCount() {
        return this.rowCount;
    }

    @Override
    public int getColumnCount() {
        return this.columnCount;
    }

    @Override
    @SuppressWarnings("PMD.MethodReturnsInternalArray")
    // Justification: the whole purpose of this method is to provide direct
    // access to the internal board, to allow users of this method to implement
    // their own logic with as little overhead as possible. 
    // One possible use case could include creating bitmaps directly from this
    // integer array, for example.
    // The documentation of this method literally states that it should be used
    // with care and it has "unsafe" in it's name, so yeah, by contract callers
    // shouldn't change random things on the internal board :)
    public int[] getBoardStateUnsafe() {
        return this.board;
    }
    
    @Override
    public int getMinimumSolutionSize() {
        return this.solutionSize;
    }

    @Override
    public IC4BoardSlim deepCopy() {
        return new C4BoardSlim(this);
    }
    
    @Override
    public void updateTokenUnsafe(final int rowIndex, final int columnIndex, final int value) {
        // this is a primarily internal API, so no custom bounds checks (performance)
        // for example this API allows indexing out of bounds columns (as long as the row index
        // isn't too high)
        // also: java does bounds checks automatically anyways, so it's not like we're risking
        // buffer overflows or raw memory access here :)
        this.board[rowIndex * this.columnCount + columnIndex] = value;
    }

    @Override
    public IC4Player getPlayerByToken(final int fieldState) {
        return this.players.getValueOrDefault(fieldState);
    }

    @Override
    public int getTokenUnsafe(final int rowIndex, final int columnIndex) {
        return this.board[rowIndex * this.columnCount + columnIndex];
    }

    @Override
    public IC4SolutionSlim tryFindFirstSolution(final boolean updateCache) {
        return this.evaluator.tryFindFirstSolution(updateCache);
    }

    @Override
    public IC4SolutionSlim tryFindFirstSolution() {
        return this.evaluator.tryFindFirstSolution(true);
    }

    @Override
    public Set<IC4SolutionSlim> findAllSolutions(final boolean updateCache) {
        return this.evaluator.findAllSolutions(updateCache);
    }

    @Override
    public Set<IC4SolutionSlim> findAllSolutions() {
        return this.evaluator.findAllSolutions(true);
    }

    @Override
    public void resetSolutionAnalyzerCaches() {
        this.evaluator.resetAnalyzerCaches();
    }

    @Override
    public boolean isEmptyUnsafe(final int row, final int column) {
        return getTokenUnsafe(row, column) == EMPTY_TOKEN;
    }

    @Override
    public boolean isSolidUnsafe(final int row, final int column) {
        if (this.checkBounds(row, column)) {
            return !this.isEmptyUnsafe(row, column);
        } else {
            // the ground is also solid
            return this.checkBounds(row - 1, column);
        }
    }

    @Override
    public boolean checkBounds(final int row, final int column) {
        // full check here (including lower bounds)
        return row >= 0 && row < this.rowCount && column >= 0 && column < this.columnCount;
    }

    @Override
    public int getDematPositionsByTokenUnsafe(final long[] buffer, final int token) {
        int index = 0;
        for (int row = 0; row < this.rowCount; row++) {
            for (int col = 0; col < this.columnCount; col++) {
                if (this.getTokenUnsafe(row, col) == token) {
                    // mitigate need for heap allocation by dematerialization to simple value type
                    buffer[index] = C4PositionMaterializer.dematerialize(row, col);
                    index++;
                }
            }
        }
        return index;
    }

    @Override
    public int getLegalDematPositionsUnsafe(long[] buffer) {
        int index = 0;
        boolean inAir = false;
        for (int col = 0; col < this.columnCount; col++) {
            int row = this.rowCount - 1;
            for (; row >= 0 && !inAir; row--) {
                inAir = this.getTokenUnsafe(row, col) == EMPTY_TOKEN;
            }
            if (inAir) {
                inAir = false;
                // mitigate need for heap allocation by dematerialization to simple value type
                // row + 1 due to the last decrement in the loop
                buffer[index] = C4PositionMaterializer.dematerialize(row + 1, col);
                index++;
            }
        }
        return index;
    }

    @Override
    public boolean isFull() {
        for (int col = 0; col < this.columnCount; col++) {
            if (this.board[col] == EMPTY_TOKEN) {
                return false;
            }
        }
        return true;
    }

    @Override
    public String toString() {
        final StringBuilder bobTheBuilder = new StringBuilder(64);
        // Can we fix it? No, we're using Java.
        // BUT: we can at least write pretty toStrings :)
        // I mean, as pretty as an integer array is gonna get ¯\_(ツ)_/¯
        bobTheBuilder.append("C4BoardSlim[").append(this.getRowCount())
            .append('x').append(this.getColumnCount()).append("]{\n");
        for (int row = 0; row < this.getRowCount(); row++) {
            bobTheBuilder.append("  ");
            for (int col = 0; col < this.getColumnCount(); col++) {
                bobTheBuilder.append(this.getTokenUnsafe(row, col)).append(' ');
            }
            bobTheBuilder.append('\n');
        }
        bobTheBuilder.append('}');
        return bobTheBuilder.toString();
    }
}
