package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;

/**
 * The default implementation of {@link IC4Field}.
 */
public class C4Field implements IC4Field {

    /**
     * The parent board that contains this field.
     */
    private final C4Board board;

    /**
     * The row of this field.
     */
    private final int row;

    /**
     * The column of this field.
     */
    private final int column;

    /**
     * The lazily initialized {@link IC4Position} instance that represents the position of this field.
     */
    private IC4Position position;

    /**
     * Creates a new instance of {@link C4Field}.
     * 
     * @param board The parent board that contains this field.
     * @param row The row of this field.
     * @param column The column of this field.
     */
    public C4Field(final C4Board board, final int row, final int column) {
        this.board = board;
        this.row = row;
        this.column = column;
    }

    /**
     * Creates a new instance of {@link C4Field}.
     * 
     * @param board The parent board that contains this field.
     * @param position The position of this field.
     */
    public C4Field(final C4Board board, final IC4Position position) {
        this(board, position.getRow(), position.getColumn());
        this.position = position;
    }

    @Override
    public IC4Board getBoard() {
        return this.board;
    }

    @Override
    public IC4Position getBoardPosition() {
        if (this.position == null) {
            this.position = new C4Position(row, column);
        }
        return this.position;
    }

    @Override
    public Optional<IC4Player> getOccupyingPlayer() {
        final IC4BoardSlim slimBoard = this.board.getInternalBoard();
        final int token = slimBoard.getTokenUnsafe(this.row, this.column);
        if (token == slimBoard.emptyToken()) {
            return Optional.empty();
        }
        return Optional.of(slimBoard.getPlayerByToken(token));
    }

    @Override
    public boolean trySetOccupyingPlayer(final IC4Player player, final boolean allowOverride) {
        final IC4BoardSlim slimBoard = this.board.getInternalBoard();
        final int token = slimBoard.getTokenUnsafe(this.row, this.column);
        if (!allowOverride && token != slimBoard.emptyToken()) {
            return false;
        }
        // row + 1 because row 0 is top row
        final int rowBeneath = this.row + 1; 
        // ensure we don't violate the laws on physics
        if (slimBoard.isSolidUnsafe(rowBeneath, this.column)) {
            slimBoard.updateTokenUnsafe(this.row, this.column, player.getToken());
            return true;
        }
        return false;
    }

    @Override
    public boolean hasNeighbor(final C4Direction direction) {
        final IC4Position neighborPosition = direction.stepFrom(this.getBoardPosition(), 1);
        return this.board.getInternalBoard().checkBounds(neighborPosition.getRow(), neighborPosition.getColumn());
    }

    @Override
    public IC4Field getNeighbor(final C4Direction direction) {
        return getNeighborInternal(direction, true);
    }

    @Override
    public Optional<IC4Field> tryGetNeighbor(final C4Direction direction) {
        final IC4Field neighbor = getNeighborInternal(direction, false);
        return Optional.ofNullable(neighbor);
    }

    /**
     * Gets the neighbor of this field in the provided direction.
     * 
     * @param direction The direction to get the neighbor in.
     * @param throwOob Whether to throw an {@link IndexOutOfBoundsException} if the neighbor is 
     * out of bounds, or to return null.
     * @return The neighbor of this field in the provided direction, or null if the neighbor is 
     * out of bounds and throwOob is false.
     */
    private IC4Field getNeighborInternal(final C4Direction direction, final boolean throwOob) {
        final IC4Position neighborPosition = direction.stepFrom(this.getBoardPosition(), 1);
        final Map<IC4Position, IC4Field> fieldCache = this.board.getFieldCache();
        IC4Field neighbor = fieldCache.get(neighborPosition);
        if (neighbor != null) {
            return neighbor;
        }
        
        if (!this.board.checkBounds(neighborPosition.getRow(), neighborPosition.getColumn())) {
            if (throwOob) {
                throw new IndexOutOfBoundsException("The provided direction violates the bounds of the game board.");
            }
            return null;
        }
        neighbor = new C4Field(board, neighborPosition);
        fieldCache.put(neighborPosition, neighbor);
        return neighbor;
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IC4Field)) {
            return false;
        }
        final IC4Field other = (IC4Field) obj;
        return this.getBoardPosition().equals(other.getBoardPosition()) 
            && this.getBoard().equals(other.getBoard());
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = hash * 31 + this.getBoardPosition().hashCode();
        hash = hash * 31 + this.getBoard().hashCode();
        return hash;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder(32);
        sb.append("C4Field[player=").append(this.getOccupyingPlayer().map(IC4Player::getName).orElse("empty"))
            .append(", position=").append(this.getBoardPosition())
            .append(']');
        return sb.toString();
    }

    @Override
    public IC4Field deepCopy() {
        return new C4Field(this.board, this.row, this.column);
    }
}
