package de.fhdw.gaming.ipspiel23.c4.domain.impl.evaluation;

import java.util.HashSet;
import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim;

/**
 * A state-less class that evaluates a given board for solutions.
 * <p>
 * Note: This is an internal API that may be subject to incompatible changes in future releases.
 * </p>
 */
public class C4SolutionEvaluator {
    
    /**
     * The evaluators for every direction: horizontal, vertical, diagonal left, diagonal right.
     */
    private final C4SolutionAnalyzer[] directionEvaluators;
    
    /**
     * Creates a new solution evaluator for the specified board.
     * @param board the board
     */
    public C4SolutionEvaluator(final C4BoardSlim board) {
        this.directionEvaluators = new C4SolutionAnalyzer[] {
            new C4SolutionAnalyzerHorizontal(board),
            new C4SolutionAnalyzerVertical(board),
            new C4SolutionAnalyzerDiagonalRight(board),
            new C4SolutionAnalyzerDiagonalLeft(board)
        };
    }
    
    /**
     * Resets the analyzer cache.
     */
    public void resetAnalyzerCaches() {
        for (final C4SolutionAnalyzer analyzer : directionEvaluators) {
            analyzer.resetCache();
        }
    }

    /**
     * Lazily identifies the first solution, if any.
     * @param updateCache Whether to update the solution cache, preventing unsolvable lines from being scanned again.
     * @return the first solution or {@code null} if no solution was found
     * @apiNote This is an internal API that may be subject to incompatible changes in future releases.
     */
    public IC4SolutionSlim tryFindFirstSolution(final boolean updateCache) {
        IC4SolutionSlim solution = null;
        for (final C4SolutionAnalyzer analyzer : this.directionEvaluators) {
            solution = analyzer.tryFindFirstSolution(solution, updateCache);
        }
        return solution;
    }
    
    /**
     * Eagerly identifies all solutions on the board.
     * @param updateCache Whether to update the solution cache, preventing unsolvable lines from being scanned again.
     * @return the set of all solutions.
     * @apiNote This is an internal API that may be subject to incompatible changes in future releases.
     */
    public Set<IC4SolutionSlim> findAllSolutions(final boolean updateCache) {
        final Set<IC4SolutionSlim> solutions = new HashSet<>();
        for (final C4SolutionAnalyzer analyzer : this.directionEvaluators) {
            analyzer.findAllSolutions(solutions, updateCache);
        }
        return solutions;
    }
}
