package de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayer;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgState;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.move.AnswerOptions;

/**
 * Implements {@link FzgState}.
 * 
 *
 */
public class FzgStateImpl implements FzgState {

    /**
     * First Player.
     */
    private final FzgPlayer firstPlayer;

    /**
     * Second Player.
     */
    private final FzgPlayer secondPlayer;

    /**
     * Contructor, creates a FzgState.
     * 
     * @param firstPlayer  The first player.
     * @param secondPlayer The second player.
     * @throws GameException if the state cannot be created according to the rules of the game.
     */
    public FzgStateImpl(final FzgPlayer firstPlayer, final FzgPlayer secondPlayer) {
        this.firstPlayer = firstPlayer;
        this.secondPlayer = secondPlayer;

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Used for DeepCopy.
     * 
     * @param source state to copy.
     */
    FzgStateImpl(final FzgStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy();
        this.secondPlayer = source.secondPlayer.deepCopy();
    }

    @Override
    public Map<String, FzgPlayer> getPlayers() {
        final Map<String, FzgPlayer> retVal = new HashMap<>();
        retVal.put(this.firstPlayer.getName(), this.firstPlayer);
        retVal.put(this.secondPlayer.getName(), this.secondPlayer);
        return retVal;
    }

    @Override
    public String toString() {
        return String.format("FzgState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer, this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof FzgStateImpl) {
            final FzgStateImpl other = (FzgStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    /**
     * Getter fristPlayer.
     * 
     * @return the firstPlayer
     */
    @Override
    public FzgPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Getter secondPlayer.
     * 
     * @return the secondPlayer
     */
    @Override
    public FzgPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public Set<FzgPlayer> computeNextPlayers() {
        final HashSet<FzgPlayer> retVal = new HashSet<>();
        if (this.firstPlayer.getAnswer().isEmpty()) {
            retVal.add(this.firstPlayer);
        }
        if (this.secondPlayer.getAnswer().isEmpty()) {
            retVal.add(this.secondPlayer);
        }
        return retVal;
    }

    @Override
    public void nextTurn() {
        final Set<FzgPlayer> playersWithoutMove = this.computeNextPlayers();
        if (playersWithoutMove.isEmpty()) {
            final AnswerOptions answerOfFirstPlayer = this.firstPlayer.getAnswer().orElseThrow();
            final AnswerOptions answerOfSecondPlayer = this.secondPlayer.getAnswer().orElseThrow();

            final Double outcomeOfFirstPlayer = this.firstPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.firstPlayer.setState(outcomeToState(outcomeOfFirstPlayer));
            this.firstPlayer.setOutcome(outcomeOfFirstPlayer);

            final Double outcomeOfSecondPlayer = this.secondPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.secondPlayer.setState(outcomeToState(outcomeOfSecondPlayer));
            this.secondPlayer.setOutcome(outcomeOfSecondPlayer);
        }

    }

    @Override
    public FzgState deepCopy() {
        return new FzgStateImpl(this);
    }

    /**
     * Converts outcome to playerState.
     * 
     * @param outcome the players outcome.
     */
    private PlayerState outcomeToState(final Double outcome) {
        // Das Spiel hat kein Verlierer, weil die beiden Spiler, laut Aufgabenstellung, bei einem nicht zustande
        // kommenden Date dennoch mögen.
        // Daher wurde entschieden, dass ein Spieler immer den State DRAW hat, es sei denn, er/sie hat eine positive
        // Auszahlung.
        return outcome > 0.0 ? PlayerState.WON : PlayerState.DRAW;
    }
}
