/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel23-tictactoe-gui.
 *
 * ipspiel23-tictactoe-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel23-tictactoe-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel23-tictactoe-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel23.tictactoe.gui.impl;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.gui.GuiObserver;
import de.fhdw.gaming.ipspiel23.tictactoe.core.domain.TicTacToeGame;
import de.fhdw.gaming.ipspiel23.tictactoe.gui.TicTacToeBoardEventProvider;
import javafx.scene.Node;

/**
 * Implements {@link GuiObserver}.
 */
final class TicTacToeGuiObserverImpl implements GuiObserver {

    /**
     * The primary (i.e. first created) {@link TicTacToeGuiObserverImpl} instance of this class.
     */
    private static final AtomicReference<WeakReference<TicTacToeGuiObserverImpl>> INSTANCE = new AtomicReference<>();

    /**
     * The {@link TicTacToeBoardView} objects per Tic Tac Toe game ID.
     */
    private final Map<Integer, TicTacToeBoardView> boardViews;

    /**
     * Creates an {@link TicTacToeGuiObserverImpl}.
     */
    TicTacToeGuiObserverImpl() {
        TicTacToeGuiObserverImpl.INSTANCE.compareAndSet(null, new WeakReference<>(this));
        this.boardViews = new LinkedHashMap<>();
    }

    /**
     * Returns a {@link TicTacToeBoardEventProvider} for a given game.
     *
     * @param gameId The game ID.
     * @return The {@link TicTacToeBoardEventProvider}.
     */
    static Optional<TicTacToeBoardEventProvider> getEventProvider(final int gameId) {
        final TicTacToeGuiObserverImpl instance = Optional.ofNullable(TicTacToeGuiObserverImpl.INSTANCE.get())
                .map(Reference::get).orElse(null);
        if (instance == null) {
            return Optional.empty();
        } else {
            final TicTacToeBoardView boardView = instance.boardViews.get(gameId);
            return boardView == null ? Optional.empty() : Optional.of(new TicTacToeBoardEventProviderImpl(boardView));
        }
    }

    @Override
    public Optional<Node> gameCreated(final Game<?, ?, ?, ?> game) {
        if (game instanceof TicTacToeGame) {
            final TicTacToeBoardView boardView = new TicTacToeBoardView((TicTacToeGame) game);
            this.boardViews.put(game.getId(), boardView);
            return Optional.of(boardView.getNode());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public void gamePaused(final Game<?, ?, ?, ?> game) {
        if (game instanceof TicTacToeGame) {
            final TicTacToeBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gamePaused((TicTacToeGame) game);
            }
        }
    }

    @Override
    public void gameResumed(final Game<?, ?, ?, ?> game) {
        if (game instanceof TicTacToeGame) {
            final TicTacToeBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gameResumed((TicTacToeGame) game);
            }
        }
    }

    @Override
    public void gameDestroyed(final Game<?, ?, ?, ?> game) {
        if (game instanceof TicTacToeGame) {
            final TicTacToeBoardView boardView = this.boardViews.remove(game.getId());
            if (boardView != null) {
                boardView.destroy((TicTacToeGame) game);
            }
        }
    }
}
