
package de.fhdw.gaming.ipspiel24.fg.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel24.fg.domain.FGActivity;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayer;
import de.fhdw.gaming.ipspiel24.fg.domain.FGState;

/**
 * Implements {@link FGState}.
 */
final class FGStateImpl implements FGState {

    /**
     * The first player.
     */
    private final FGPlayer firstPlayer;
    /**
     * The second player.
     */
    private final FGPlayer secondPlayer;

    /**
     * Creates a FG state.
     *
     * @param firstPlayer  The first player.
     * @param secondPlayer The second player.
     * @throws GameException if the state cannot be created according to the rules of the game.
     */
    FGStateImpl(final FGPlayer firstPlayer, final FGPlayer secondPlayer)
            throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayer, "firstPlayer");
        this.secondPlayer = Objects.requireNonNull(secondPlayer, "secondPlayer");

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a FG state by copying an existing one.
     *
     * @param source The state to copy.
     */
    FGStateImpl(final FGStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy();
        this.secondPlayer = source.secondPlayer.deepCopy();
    }

    /**
     * Returns the first player.
     */
    @Override
    public FGPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public FGPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("FGState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer, this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof FGStateImpl) {
            final FGStateImpl other = (FGStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public FGState deepCopy() {
        return new FGStateImpl(this);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, FGPlayer> getPlayers() {
        final Map<String, FGPlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public Set<FGPlayer> computeNextPlayers() {
        final Set<FGPlayer> playersWithoutMove = new LinkedHashSet<>();
        if (this.firstPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.firstPlayer);
        }
        if (this.secondPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.secondPlayer);
        }
        return playersWithoutMove;
    }

    @Override
    public void nextTurn() {
        final Set<FGPlayer> playersWithoutMove = this.computeNextPlayers();
        if (playersWithoutMove.isEmpty()) {
            final FGActivity answerOfFirstPlayer = this.firstPlayer.getAnswer().orElseThrow();
            final FGActivity answerOfSecondPlayer = this.secondPlayer.getAnswer().orElseThrow();

            final Double outcomeOfFirstPlayer = this.firstPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.firstPlayer.setState(outcomeToState(outcomeOfFirstPlayer));
            this.firstPlayer.setOutcome(outcomeOfFirstPlayer);

            final Double outcomeOfSecondPlayer = this.secondPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.secondPlayer.setState(outcomeToState(outcomeOfSecondPlayer));
            this.secondPlayer.setOutcome(outcomeOfSecondPlayer);
        }
    }

    /**
     * Computes a player state from an outcome.
     *
     * @param outcome The player's outcome.
     */
    private static PlayerState outcomeToState(final Double outcome) {
        return outcome > 0.0 ? PlayerState.WON : outcome < 0.0 ? PlayerState.LOST : PlayerState.DRAW;
    }
}
