/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-VierConnects-gui.
 *
 * ipspiel24-VierConnects-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel24-VierConnects-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel24-VierConnects-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.VierConnects.gui.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsBoard;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsField;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsFieldState;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsPlayer;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsPosition;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsState;
import de.fhdw.gaming.ipspiel24.VierConnects.gui.VierConnectsBoardEventProvider;
import de.fhdw.gaming.ipspiel24.VierConnects.gui.event.VierConnectsBoardEvent;
import de.fhdw.gaming.ipspiel24.VierConnects.gui.event.VierConnectsMakeMoveBoardEvent;
import javafx.application.Platform;
import javafx.scene.Cursor;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

/**
 * Implements user interaction with a VierConnects board.
 */
final class VierConnectsBoardEventProviderImpl implements VierConnectsBoardEventProvider {

    /**
     * The associated {@link VierConnectsBoardView}.
     */
    private final VierConnectsBoardView boardView;

    /**
     * Constructor.
     *
     * @param boardView The associated {@link VierConnectsBoardView}.
     */
    VierConnectsBoardEventProviderImpl(final VierConnectsBoardView boardView) {
        this.boardView = boardView;
    }

    @Override
    public VierConnectsBoardEvent waitForEvent(final VierConnectsPlayer player, final VierConnectsState state) {
        final Map<VierConnectsPosition, VierConnectsFieldView> emptyFieldViews = new LinkedHashMap<>();
        for (final Map.Entry<VierConnectsPosition, ? extends VierConnectsField> entry : state.getBoard()
                .getFieldsBeing(VierConnectsFieldState.EMPTY).entrySet()) {
            emptyFieldViews.put(entry.getKey(), this.boardView.getFieldView(entry.getKey()).orElseThrow());
        }

        final AtomicReference<VierConnectsBoardEvent> event = new AtomicReference<>();
        final Runnable cleanUp;
        if (emptyFieldViews.isEmpty()) {
            final List<VierConnectsFieldView> fieldViews = this.setupInactiveFields(state.getBoard());
            cleanUp = () -> this.cleanUpFields(fieldViews);
        } else {
            this.setupActiveFields(emptyFieldViews, event);
            cleanUp = () -> this.cleanUpFields(emptyFieldViews.values());
        }

        try {
            this.boardView.getUserInputSemaphore().acquire();
            return event.get();
        } catch (final InterruptedException e) {
            return null;
        } finally {
            Platform.runLater(cleanUp);
        }
    }

    /**
     * Sets up the fields when a move is possible (and hence the set of empty fields is not empty).
     *
     * @param emptyFieldViews The non-empty set of views for active fields.
     * @param event           The event to be set when the user selects an active field.
     */
    private void setupActiveFields(final Map<VierConnectsPosition, VierConnectsFieldView> emptyFieldViews,
            final AtomicReference<VierConnectsBoardEvent> event) {
        for (final Map.Entry<VierConnectsPosition, VierConnectsFieldView> entry : emptyFieldViews.entrySet()) {
            final VierConnectsPosition position = entry.getKey();
            final VierConnectsFieldView fieldView = entry.getValue();
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CROSSHAIR);
                fieldView.setHighlighted(true);
                fieldView.setOnMouseClicked((final MouseEvent mouseEvent) -> {
                    if (mouseEvent.getButton() == MouseButton.PRIMARY) {
                        event.set(new VierConnectsMakeMoveBoardEvent(position));
                        this.boardView.getUserInputSemaphore().release();
                    }
                });
            });
        }
    }

    /**
     * Sets up the fields when no move is possible (and hence the set of empty fields is empty).
     *
     * @param board The Vier Connects board.
     * @return A list of all field views on the board.
     */
    private List<VierConnectsFieldView> setupInactiveFields(final VierConnectsBoard board) {
        final List<VierConnectsField> fields = new ArrayList<>();
        board.getFields().forEach((final List<? extends VierConnectsField> list) -> fields.addAll(list));

        final List<VierConnectsFieldView> fieldViews = new ArrayList<>();
        for (final VierConnectsField field : fields) {
            final VierConnectsFieldView fieldView = this.boardView.getFieldView(field.getPosition()).orElseThrow();
            fieldViews.add(fieldView);
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CLOSED_HAND);
                fieldView.setOnMouseClicked(null);
            });
        }

        return fieldViews;
    }

    /**
     * Cleans up after user interaction by resetting mouse cursors and removing event handlers.
     *
     * @param fieldViews The modified field views.
     */
    private void cleanUpFields(final Collection<? extends VierConnectsFieldView> fieldViews) {
        for (final VierConnectsFieldView fieldView : fieldViews) {
            fieldView.setCursor(Cursor.DEFAULT);
            fieldView.setHighlighted(false);
            fieldView.setOnMouseClicked(null);
        }
    }

    @Override
    public void cancelWaiting() {
        this.boardView.getUserInputSemaphore().release();
    }
}
