/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-VierConnects-gui.
 *
 * ipspiel24-VierConnects-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel24-VierConnects-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel24-VierConnects-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.VierConnects.gui.impl;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsPlayer;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsState;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsStrategy;
import de.fhdw.gaming.ipspiel24.VierConnects.core.moves.VierConnectsMove;
import de.fhdw.gaming.ipspiel24.VierConnects.core.moves.factory.VierConnectsMoveFactory;
import de.fhdw.gaming.ipspiel24.VierConnects.gui.VierConnectsBoardEventProvider;

/**
 * Implements {@link VierConnectsStrategy} by asking the user for a move.
 */
public final class VierConnectsInteractiveStrategy implements VierConnectsStrategy {

    /**
     * The factory for creating VierConnects moves.
     */
    private final VierConnectsMoveFactory moveFactory;

    /**
     * Creates an {@link VierConnectsInteractiveStrategy}.
     *
     * @param moveFactory The factory for creating VierConnects moves.
     */
    VierConnectsInteractiveStrategy(final VierConnectsMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    /**
     * Asks the user to choose a move.
     */
    @Override
    public Optional<VierConnectsMove> computeNextMove(final int gameId, final VierConnectsPlayer player,
            final VierConnectsState state, final long maxComputationTimePerMove) throws GameException {

        final Optional<VierConnectsBoardEventProvider> provider = VierConnectsGuiObserverImpl.getEventProvider(gameId);
        if (provider.isEmpty()) {
            return Optional.empty();
        }

        final AtomicReference<VierConnectsMove> move = new AtomicReference<>();
        provider.get().waitForEvent(player, state).accept(event -> move.setPlain(
                this.moveFactory.createPlaceMarkMove(player.isUsingCrosses(), event.getFieldPosition().getColumn())));

        return Optional.ofNullable(move.getPlain());
    }

    @Override
    public String toString() {
        return VierConnectsInteractiveStrategy.class.getSimpleName();
    }

    @Override
    public boolean isInteractive() {
        return true;
    }

    @Override
    public void abortRequested(final int gameId) {
        final Optional<VierConnectsBoardEventProvider> provider = VierConnectsGuiObserverImpl.getEventProvider(gameId);
        if (!provider.isEmpty()) {
            provider.get().cancelWaiting();
        }
    }
}
