/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-demo.
 *
 * Ipspiel24-demo is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel24-demo is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel24-demo. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultGame;
import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Observer;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfGameBuilder;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfPlayer;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfPlayerBuilder;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfState;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfStrategy;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.moves.MuenzwurfMove;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.moves.impl.AbstractMuenzwurfMove;

/**
 * Implements {@link MuenzwurfGameBuilder}.
 */
final class MuenzwurfGameBuilderImpl implements MuenzwurfGameBuilder {

    /**
     * The {@link Observer}s to be attached to the game.
     */
    private final List<Observer> observers;
    /**
     * The player using black tokens.
     */
    private Optional<MuenzwurfPlayer> firstPlayer;
    /**
     * The strategy of the player using black tokens.
     */
    private Optional<MuenzwurfStrategy> firstPlayerStrategy;
    /**
     * The player using white tokens.
     */
    private Optional<MuenzwurfPlayer> secondPlayer;
    /**
     * The strategy of the player using white tokens.
     */
    private Optional<MuenzwurfStrategy> secondPlayerStrategy;
    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a Muenzwurf game builder.
     */
    MuenzwurfGameBuilderImpl() {
        this.observers = new ArrayList<>();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public MuenzwurfPlayerBuilder createPlayerBuilder() {
        return new MuenzwurfPlayerBuilderImpl();
    }

    @Override
    public MuenzwurfGameBuilder addPlayer(final MuenzwurfPlayer player, final MuenzwurfStrategy strategy)
            throws GameException {

        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public MuenzwurfGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public MuenzwurfGameBuilder addObservers(final List<Observer> newObservers) {
        this.observers.addAll(newObservers);
        return this;
    }

    @Override
    public Game<MuenzwurfPlayer, MuenzwurfState, MuenzwurfMove, MuenzwurfStrategy> build(final int id)
            throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("A Muenzwurf game needs two players.");
        }

        final MuenzwurfStateImpl initialState = new MuenzwurfStateImpl(this.firstPlayer.get(), this.secondPlayer.get());

        final Map<String, MuenzwurfStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new DefaultGame<>(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractMuenzwurfMove.class::isInstance,
                new MuenzwurfMoveGeneratorImpl(),
                this.observers);
    }
}
