/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-Ssp.
 *
 * Ipspiel24-Ssp is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel24-Ssp is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel24-Ssp. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.ssp.domain.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspPlayer;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspPlayerBuilder;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspPossibleChoice;

/**
 * Implements {@link SspPlayerBuilder}.
 */
final class SspPlayerBuilderImpl implements SspPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<SspPossibleChoice, Map<SspPossibleChoice, Double>>> possibleOutcomes;

    /**
     * Creates an {@link SspPlayerBuilderImpl}.
     */
    SspPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public SspPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public SspPlayerBuilder changePossibleOutcomes(
            final Map<SspPossibleChoice, Map<SspPossibleChoice, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public SspPlayer build() throws GameException {
        return new SspPlayerImpl(
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<SspPossibleChoice, Map<SspPossibleChoice, Double>> checkPossibleOutcomes(
            final Map<SspPossibleChoice, Map<SspPossibleChoice, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_ROCK, SspPossibleChoice.SPC_ROCK);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_ROCK, SspPossibleChoice.SPC_PAPER);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_ROCK, SspPossibleChoice.SPC_SCISSORS);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_PAPER, SspPossibleChoice.SPC_PAPER);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_PAPER, SspPossibleChoice.SPC_SCISSORS);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_PAPER, SspPossibleChoice.SPC_ROCK);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_SCISSORS, SspPossibleChoice.SPC_SCISSORS);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_SCISSORS, SspPossibleChoice.SPC_ROCK);
        this.checkPossibleOutcome(outcomes, SspPossibleChoice.SPC_SCISSORS, SspPossibleChoice.SPC_PAPER);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<SspPossibleChoice, Map<SspPossibleChoice, Double>> outcomes,
            final SspPossibleChoice firstChoice,
            final SspPossibleChoice secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "No outcome defined for player '%s' and combination %s/%s.",
                            this.name,
                            toAnswer(firstChoice),
                            toAnswer(secondChoice)));
        }
    }

    /**
     * Maps a boolean value to a "yes" or "no" answer.
     *
     * @param value The value to be mapped.
     */
    private static String toAnswer(final SspPossibleChoice value) {
        switch (value) {
        case SPC_ROCK:
            return "Rock";
        case SPC_PAPER:
            return "Paper";
        case SPC_SCISSORS:
            return "Scissors";
        default:
            return "default";
        }
    }
}
