package de.fhdw.gaming.ipspiel24.tictactoe.gui.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.minimax.Minimax;
import de.fhdw.gaming.ipspiel24.minimax.MinimaxStrategy;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeBoard;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeField;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeFieldState;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToePlayer;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToePosition;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeRow;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeState;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeStrategy;
import de.fhdw.gaming.ipspiel24.tictactoe.core.moves.TicTacToeMove;
import de.fhdw.gaming.ipspiel24.tictactoe.core.moves.factory.TicTacToeMoveFactory;

/**
 * Implements {@link TicTacToeStrategy} by asking the user for a move.
 */
public class TicTacToeMinimaxStrategy implements MinimaxStrategy<TicTacToePlayer, TicTacToeState, TicTacToeMove>,
        TicTacToeStrategy {

    /**
     * The factory for creating TicTacToe moves.
     */
    private final TicTacToeMoveFactory moveFactory;
    
    /**
     * Creates an {@link TicTacToeMinimaxStrategyOLD}.
     * 
     * @param moveFactory The factory for creating TicTacToe moves.
     */
    public TicTacToeMinimaxStrategy(final TicTacToeMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    @Override
    public Optional<TicTacToeMove> computeNextMove(int gameId, TicTacToePlayer player, TicTacToeState state,
            long maxComputationTimePerMove) throws GameException, InterruptedException {
        final TicTacToePlayer opponent = this.getOpponent(state, player);
        
        final Minimax<TicTacToePlayer, TicTacToeState, TicTacToeMove> minimax = 
                new Minimax<TicTacToePlayer, TicTacToeState, TicTacToeMove>(this, 10, opponent);
        final TicTacToeMove bestMove = minimax.getBestMove(state, player);
        //System.out.println("We are: " + player.toString());
        return Optional.of(bestMove);
    }

    @Override
    public List<TicTacToeMove> getPossibleMoves(final TicTacToeState state) {
        final List<TicTacToeMove> possibleMoves = new ArrayList<>();
        final TicTacToeBoard board = state.getBoard();
        final Map<TicTacToePosition,
                ? extends TicTacToeField> fieldMap = board.getFieldsBeing(TicTacToeFieldState.EMPTY);
        for (Map.Entry<TicTacToePosition, ? extends TicTacToeField> entry : fieldMap.entrySet()) {
            possibleMoves.add(moveFactory.createPlaceMarkMove(
                    state.getCurrentPlayer().isUsingCrosses(), entry.getKey()));
        }

        return possibleMoves;
    }

    @Override
    public int evaluate(TicTacToeState state, TicTacToePlayer player, int depth) {
        final TicTacToeBoard board = state.getBoard();
        final Set<TicTacToeRow> uniformRows = board.getRowsUniformlyMarked();
        if (!uniformRows.isEmpty()) {
            final TicTacToeFieldState rowState = uniformRows.iterator().next().getState().orElseThrow();
            final TicTacToeFieldState playerState = player.isUsingCrosses() 
                    ? TicTacToeFieldState.CROSS : TicTacToeFieldState.NOUGHT;
            final int returnScore = rowState.equals(playerState) ? 10 + depth : -10 - depth;
            //System.out.println("Returning score: " + returnScore + " board: " + state.getBoard());
            return returnScore;
        } /*else {
            final Map<TicTacToePosition,
                    ? extends TicTacToeField> emptyFields = board.getFieldsBeing(TicTacToeFieldState.EMPTY);
            if (emptyFields.isEmpty()) {
                return 0;
            }
        }*/
        return 0;
    }

    @Override
    public String toString() {
        return "TicTacToe Minimax Strategy";
    }

    @Override
    public TicTacToePlayer getOpponent(TicTacToeState state, TicTacToePlayer currentPlayer) {
        System.out.println("Opponent" + (state.getCurrentPlayer() == state.getCrossesPlayer() 
                ? state.getNoughtsPlayer() : state.getCrossesPlayer()) + " currentPLayer: " + currentPlayer);
        return state.getCurrentPlayer() == state.getCrossesPlayer() 
                ? state.getNoughtsPlayer() : state.getCrossesPlayer();
        
        /*final Map<String, TicTacToePlayer> players = state.getPlayers();
        for (Map.Entry<String, TicTacToePlayer> entry : players.entrySet()) {
            System.out.println("Player: " + entry.getValue().toString());
            if (entry.getValue() != currentPlayer) {
                System.out.println("Opponent: " + entry.getValue().toString());
                return entry.getValue();
            }
        }
        return null;*/
    }
}
