/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-tictactoe-strategy-minimax.
 *
 * ipspiel24-tictactoe-strategy-minimax is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel24-tictactoe-strategy-minimax is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel24-tictactoe-strategy-minimax. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.tictactoe.strategy.minimax;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.minimax.Minimax;
import de.fhdw.gaming.ipspiel24.minimax.MinimaxStrategy;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeBoard;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeField;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeFieldState;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToePlayer;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToePosition;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeRow;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeState;
import de.fhdw.gaming.ipspiel24.tictactoe.core.domain.TicTacToeStrategy;
import de.fhdw.gaming.ipspiel24.tictactoe.core.moves.TicTacToeMove;
import de.fhdw.gaming.ipspiel24.tictactoe.core.moves.factory.TicTacToeMoveFactory;

/**
 * Implements {@link TicTacToeStrategy} by asking the user for a move.
 */
public class TicTacToeMinimaxStrategy implements MinimaxStrategy<TicTacToePlayer, TicTacToeState, TicTacToeMove>,
        TicTacToeStrategy {

    /**
     * The factory for creating TicTacToe moves.
     */
    private final TicTacToeMoveFactory moveFactory;
    
    /**
     * Creates an {@link TicTacToeMinimaxStrategyOLD}.
     * 
     * @param moveFactory The factory for creating TicTacToe moves.
     */
    public TicTacToeMinimaxStrategy(final TicTacToeMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    @Override
    public Optional<TicTacToeMove> computeNextMove(int gameId, TicTacToePlayer player, TicTacToeState state,
            long maxComputationTimePerMove) throws GameException, InterruptedException {
        final TicTacToePlayer opponent = this.getOpponent(state);
        
        final Minimax<TicTacToePlayer, TicTacToeState, TicTacToeMove> minimax = 
                new Minimax<TicTacToePlayer, TicTacToeState, TicTacToeMove>(this, 10, opponent);
        final TicTacToeMove bestMove = minimax.getBestMove(state, player);
        //System.out.println("We are: " + player.toString());
        return Optional.of(bestMove);
    }

    @Override
    public List<TicTacToeMove> getPossibleMoves(final TicTacToeState state) {
        if (!state.getBoard().getRowsUniformlyMarked().isEmpty()) {
            return Collections.emptyList();
        }
        final List<TicTacToeMove> possibleMoves = new ArrayList<>();
        final TicTacToeBoard board = state.getBoard();
        final Map<TicTacToePosition,
                ? extends TicTacToeField> fieldMap = board.getFieldsBeing(TicTacToeFieldState.EMPTY);
        for (Map.Entry<TicTacToePosition, ? extends TicTacToeField> entry : fieldMap.entrySet()) {
            possibleMoves.add(moveFactory.createPlaceMarkMove(
                    state.getCurrentPlayer().isUsingCrosses(), entry.getKey()));
        }

        return possibleMoves;
    }

    @Override
    public int evaluate(TicTacToeState state, TicTacToePlayer player, int depth) {
        final TicTacToeBoard board = state.getBoard();
        final Set<TicTacToeRow> uniformRows = board.getRowsUniformlyMarked();
        if (!uniformRows.isEmpty()) {
            final TicTacToeFieldState rowState = uniformRows.iterator().next().getState().orElseThrow();
            final TicTacToeFieldState playerState = player.isUsingCrosses() 
                    ? TicTacToeFieldState.CROSS : TicTacToeFieldState.NOUGHT;
            final int returnScore = rowState.equals(playerState) ? 10 + depth : -10 - depth;
            //System.out.println("Returning score: " + returnScore + " board: " + state.getBoard());
            return returnScore;
        } /*else {
            final Map<TicTacToePosition,
                    ? extends TicTacToeField> emptyFields = board.getFieldsBeing(TicTacToeFieldState.EMPTY);
            if (emptyFields.isEmpty()) {
                return 0;
            }
        }*/
        return 0;
    }

    @Override
    public String toString() {
        return "TicTacToe Minimax Strategy";
    }

    @Override
    public TicTacToePlayer getOpponent(TicTacToeState state) {
        System.out.println("Opponent" + (state.getCurrentPlayer() == state.getCrossesPlayer() 
                ? state.getNoughtsPlayer() : state.getCrossesPlayer()) + " currentPLayer: " + state.getCurrentPlayer());
        return state.getCurrentPlayer() == state.getCrossesPlayer() 
                ? state.getNoughtsPlayer() : state.getCrossesPlayer();
        
        /*final Map<String, TicTacToePlayer> players = state.getPlayers();
        for (Map.Entry<String, TicTacToePlayer> entry : players.entrySet()) {
            System.out.println("Player: " + entry.getValue().toString());
            if (entry.getValue() != currentPlayer) {
                System.out.println("Opponent: " + entry.getValue().toString());
                return entry.getValue();
            }
        }
        return null;*/
    }
}
