package de.fhdw.gaming.ipspiel24.VierConnects.strategy.minimax;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsBoard;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsFieldState;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsPlayer;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsPosition;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsState;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsStrategy;
import de.fhdw.gaming.ipspiel24.VierConnects.core.moves.VierConnectsMove;
import de.fhdw.gaming.ipspiel24.VierConnects.core.moves.factory.VierConnectsMoveFactory;
import de.fhdw.gaming.ipspiel24.minimax.Minimax;
import de.fhdw.gaming.ipspiel24.minimax.MinimaxStrategy;

/**
 * Minimax Strategy for VierConnects.
 */
public class VierConnectsMinimaxStrategy implements VierConnectsStrategy,
        MinimaxStrategy<VierConnectsPlayer, VierConnectsState, VierConnectsMove> {

    /**
     * the moveFactory.
     */
    private final VierConnectsMoveFactory moveFactory;

    /**
     * maxDepth of Search Tree.
     */
    private final int maxDepth = 5;

    /**
     * multiplier for centre positions.
     */
    private final int scoreCenterCountMultiplier = 2;

    /**
     * score for player evaluation: win.
     */
    private final int scoreWin = 999999;

    /**
     * score for player evaluation: 3 in a row and 1 empty.
     */
    private final int score3Row = 100;

    /**
     * score for player evaluation: 2 in a row and 2 empty.
     */
    private final int score2Row = 50;

    // these need to be positive numbers, as they get subtracted from the score.
    /**
     * score for enemy evaluation: win.
     */
    private final int scoreLose = 99999;

    /**
     * score for enemy evaluation: 3 in a row and 1 empty.
     */
    private final int score3RowEnemy = 100;

    /**
     * constructor.
     * 
     * @param moveFactory public moveFactory.
     */
    public VierConnectsMinimaxStrategy(VierConnectsMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    @Override
    public Optional<VierConnectsMove> computeNextMove(int gameId, VierConnectsPlayer player, VierConnectsState state,
            long maxComputationTimePerMove) throws GameException, InterruptedException {
        // long startTime = System.nanoTime(); //left these in case i want to measure time again for something
        final VierConnectsPlayer opponent = this.getOpponent(state);
        final Minimax<VierConnectsPlayer, VierConnectsState,
                VierConnectsMove> minimax = new Minimax<VierConnectsPlayer, VierConnectsState, VierConnectsMove>(this,
                        this.maxDepth, opponent);
        final VierConnectsMove bestMove = minimax.getBestMove(state, player);
        // long endTime = System.nanoTime();
        // System.out.print((endTime - startTime) + ";");
        return Optional.of(bestMove);
    }

    @Override
    public List<VierConnectsMove> getPossibleMoves(VierConnectsState state) {

        final List<VierConnectsMove> possibleMoves = new ArrayList<>();
        final VierConnectsBoard board = state.getBoard();
        for (int i = 0; i < board.getColumnSize(); i++) {
            final VierConnectsPosition pos = VierConnectsPosition.of(0, i);
            if (board.getFieldAt(pos).getState() == VierConnectsFieldState.EMPTY) {
                possibleMoves.add(moveFactory.createPlaceMarkMove(
                        state.getCurrentPlayer().isUsingCrosses(), i));
            }
        }

        return orderMoves(possibleMoves);
    }

    /**
     * takes possibleMoves in a List and orders it to start from the middle.
     * e.g.: 0,1,2,3,4,5,6,7 -> 3,4,2,5,1,6,0,7
     * (prioritise better Moves for more effective alpha- beta-pruning)
     * 
     * @param possibleMoves list of ordered possibleMoves left to right
     * @return new ordered list of possible Moves starting from the beginning.
     */
    private List<VierConnectsMove> orderMoves(List<VierConnectsMove> possibleMoves) {
        final List<VierConnectsMove> orderedMoves = new ArrayList<>(possibleMoves.size());
        final int middle = possibleMoves.size() / 2;

        for (int i = 0; i < possibleMoves.size(); i++) {
            final int index = middle + (i % 2 == 0 ? i / 2 : -(i / 2 + 1));
            orderedMoves.add(possibleMoves.get(index));
        }

        return orderedMoves;
    }

    /**
     * evaluates window of four positions and gives score for 2,3,4 in a row.
     * also gives bas score for good opponent position.
     * 
     * @param window      array of four fields (the state of the fields)
     * @param playerState the state of the player that is using minimax
     * @return score of window.
     */
    @SuppressWarnings("checkstyle:CyclomaticComplexity")
    private int evaluateWindow(VierConnectsFieldState[] window, VierConnectsFieldState playerState) {
        int score = 0;

        final VierConnectsFieldState opponent = this.getOpponent(playerState);

        final long playerCount = Arrays.stream(window).filter(p -> p == playerState).count();
        final long opponentCount = Arrays.stream(window).filter(p -> p == opponent).count();
        final long emptyCount = Arrays.stream(window).filter(p -> p == VierConnectsFieldState.EMPTY).count();

        if (playerCount == 4) {
            score += this.scoreWin;
        } else if (playerCount == 3 && emptyCount == 1) {
            score += this.score3Row;
        } else if (playerCount == 2 && emptyCount == 2) {
            score += this.score2Row;
        }

        if (opponentCount == 4) {
            score -= this.scoreLose;
        } else if (opponentCount == 3 && emptyCount == 1) {
            score -= this.score3RowEnemy;
        }

        return score;
    }

    /**
     * returns a score for all rows or columns.
     * 
     * @param board       board Board of the current State that gets evaluated.
     * @param playerState State of the current Player
     * @param isColumn    boolean whether rows or columns get evaluated.
     * @return score for all rows or columns
     */
    private int evaluateDirection(VierConnectsBoard board, VierConnectsFieldState playerState, boolean isColumn) {
        int score = 0;
        final int primarySize = isColumn ? board.getColumnSize() : board.getRowSize();
        final int secondarySize = isColumn ? board.getRowSize() : board.getColumnSize();

        for (int primary = 0; primary < primarySize; primary++) {
            final VierConnectsFieldState[] array = new VierConnectsFieldState[secondarySize];
            for (int secondary = 0; secondary < secondarySize; secondary++) {
                final VierConnectsPosition position = isColumn ? VierConnectsPosition.of(secondary, primary)
                        : VierConnectsPosition.of(primary, secondary);
                array[secondary] = board.getFieldAt(position).getState();
            }
            for (int secondary = 0; secondary < secondarySize - 3; secondary++) {
                final VierConnectsFieldState[] window = Arrays.copyOfRange(array, secondary, secondary + 4);
                score += evaluateWindow(window, playerState);
            }
        }

        return score;
    }

    /**
     * returns a score for all (positive or negative) diagonal positions.
     * 
     * @param board           Board of the current State that gets evaluated.
     * @param playerState     State of the current Player
     * @param isPositiveSlope boolean whether the negative or positive slopes get evaluated.
     * @return score for positive or negative positions.
     */
    private int evaluateDiagonals(VierConnectsBoard board, VierConnectsFieldState playerState,
            boolean isPositiveSlope) {
        int score = 0;
        final int rowSize = board.getRowSize();
        final int columnSize = board.getColumnSize();

        for (int r = 0; r < rowSize - 3; r++) {
            for (int c = 0; c < columnSize - 3; c++) {
                final VierConnectsFieldState[] window = new VierConnectsFieldState[4];
                for (int i = 0; i < 4; i++) {
                    if (isPositiveSlope) {
                        window[i] = board.getFieldAt(VierConnectsPosition.of(r + i, c + i)).getState();
                    } else {
                        window[i] = board.getFieldAt(VierConnectsPosition.of(r + 3 - i, c + i)).getState();
                    }
                }
                score += evaluateWindow(window, playerState);
            }
        }

        return score;
    }

    @Override
    public int evaluate(VierConnectsState state, VierConnectsPlayer player, int depth) {
        int score = 0;

        final VierConnectsBoard board = state.getBoard();
        final int columnSize = board.getColumnSize();
        final int rowSize = board.getRowSize();
        final VierConnectsFieldState playerState = player.isUsingCrosses() ? VierConnectsFieldState.CROSS
                : VierConnectsFieldState.NOUGHT;

        // score for middle position
        int centerCount = 0;
        for (int r = 0; r < rowSize; r++) {
            if (board.getFieldAt(VierConnectsPosition.of(r, columnSize / 2)).getState() == playerState) {
                centerCount++;
            }
        }
        score += centerCount * this.scoreCenterCountMultiplier;
        // score for depth
        score += this.maxDepth - depth;

        // Evaluate columns
        score += evaluateDirection(board, playerState, true);

        // Evaluate rows
        score += evaluateDirection(board, playerState, false);

        // Evaluate positive sloped diagonals
        score += evaluateDiagonals(board, playerState, true);

        // Evaluate negative sloped diagonals
        score += evaluateDiagonals(board, playerState, false);

        return score;
    }

    @Override
    public VierConnectsPlayer getOpponent(VierConnectsState state) {
        return state.getCurrentPlayer() == state.getCrossesPlayer()
                ? state.getNoughtsPlayer()
                : state.getCrossesPlayer();
    }

    /**
     * returns the opponent FieldState for a given VierConnectsFieldState.
     * 
     * @param state FieldState of the current Player.
     * @return opposite State of the given fieldState.
     */
    public VierConnectsFieldState getOpponent(VierConnectsFieldState state) {
        return state.equals(VierConnectsFieldState.CROSS) ? VierConnectsFieldState.NOUGHT
                : VierConnectsFieldState.CROSS;
    }

    @Override
    public String toString() {
        return "Vier-Gewinnt Minimax Strategy";
    }

}
