package de.fhdw.gaming.ipspiel24.minimax;

import java.util.List;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.State;

/**
 * Minimax Class, needs to be implemented by Strategy.
 *
 * @param <P> Generic Player, special Player needs to be passed at implementation
 * @param <S> Generic State, special State needs to be passed at implementation
 * @param <M> Generic Move, special Move needs to be passed at implementation
 */
public class Minimax<P extends Player<P>, S extends State<P, S>, M extends Move<P, S>> {

    /**
     * Max depth of Minimax Tree.
     */
    private final int maxDepth;
    /**
     * Strategy that implements MinimaxStrategy.
     */
    private final MinimaxStrategy<P, S, M> strategy;
    /**
     * Opponent of player that uses minimax.
     */
    private final P opponent;

    /**
     * score where abs(score) > endScore means definitive End of Game. e.g. a win.
     */
    private final int endScore = 10000;

    /**
     * Minimax class.
     *
     * @param strategy MinimaxStrategy that is implemented. Pass instance.
     * @param maxDepth maximum depth of minimax tree, larger values = move computation effort.
     * @param opponent opponent of minimax player
     */
    public Minimax(final MinimaxStrategy<P, S, M> strategy, final int maxDepth, final P opponent) {
        this.strategy = strategy;
        this.maxDepth = maxDepth;
        this.opponent = opponent;
    }

    /**
     * returns best move for given game state.
     *
     * @param state  State of the game.
     * @param player Active Player.
     */
    public M getBestMove(final S state, final P player) throws GameException {

        final List<M> possibleMoves = this.strategy.getPossibleMoves(state);
        if (possibleMoves.isEmpty()) {
            return null;
        }

        int bestScore = Integer.MIN_VALUE + 1;
        M bestMove = null;
        for (final M move : possibleMoves) {
            final S stateCopy = state.deepCopy();
            move.applyTo(stateCopy, player);
            stateCopy.nextTurn();
            final int score = -this.negamax(stateCopy, player, 0, Integer.MIN_VALUE + 1, Integer.MAX_VALUE,
                    false);
            if (score > bestScore) {
                bestScore = score;
                bestMove = move;
            }
        }
        return bestMove;
    }

    /**
     * negamax algorithm.
     *
     * @param state        of game
     * @param player       player
     * @param depth        maximum depth of game tree
     * @param alpha        lower bound of range for ab-pruning
     * @param beta         upper bound for pruning
     * @param isMaximising whether the minimising player or maximising players turn
     * @return score of individual path.
     * @throws GameException
     */
    private int negamax(final S state, final P player, final int depth, final int alpha, final int beta,
            final boolean isMaximising) throws GameException {

        int newAlpha = alpha;
        final int newBeta = beta;
        final int evaluation = this.strategy.evaluate(state, player, depth);
        final List<M> possibleMoves = this.strategy.getPossibleMoves(state);

        if (depth == this.maxDepth || possibleMoves.isEmpty() || evaluation > this.endScore) {
            return evaluation * (isMaximising ? 1 : -1);
        }

        int bestValue = Integer.MIN_VALUE + 1;
        for (final M move : possibleMoves) {

            final S stateCopy = state.deepCopy();
            move.applyTo(stateCopy, isMaximising ? player : this.opponent);
            stateCopy.nextTurn();
            final int value = -this.negamax(stateCopy, player, depth + 1, -newBeta,
                    -newAlpha, !isMaximising);
            bestValue = Math.max(bestValue, value);
            newAlpha = Math.max(newAlpha, bestValue);

            if (newAlpha >= newBeta) {
                break;
            }

        }

        return bestValue;
    }

}
