package de.fhdw.wtf.context.model;

import java.math.BigInteger;

import de.fhdw.wtf.persistence.meta.IntegerValue;

/**
 * Represents integer values in the range [-10^31 up to 10^31-1].
 */
public final class Int extends AnyValue {
	
	/**
	 * The underlying value.
	 */
	private final BigInteger value;
	
	/**
	 * Creates a new Int object from a BigInteger.
	 * 
	 * @param value
	 *            The integer value.
	 */
	public Int(final BigInteger value) {
		this.value = value;
	}
	
	/**
	 * Creates a new Int object from an IntegerValue.
	 * 
	 * @param value
	 *            The integer value.
	 */
	public Int(final IntegerValue value) {
		this(value.getValue());
	}
	
	/**
	 * Creates a new Int object from a string.
	 * 
	 * @param value
	 *            The string value denoting an integer.
	 */
	public Int(final String value) {
		this(new BigInteger(value));
	}
	
	/**
	 * Creates a new Int object from a value of type long.
	 * 
	 * @param value
	 *            The integer value.
	 */
	public Int(final long value) {
		this(BigInteger.valueOf(value));
	}
	
	/**
	 * Performs the addition of two Ints.
	 * 
	 * @param summand
	 *            Another Int object which will be added.
	 * @return The sum of this Int and the other Int.
	 */
	public Int add(final Int summand) {
		return new Int(this.value.add(summand.value));
	}
	
	/**
	 * Performs the subtraction of two Ints.
	 * 
	 * @param subtrahend
	 *            Another Int object which will be subtracted.
	 * @return The difference of this Int and the other Int.
	 */
	public Int sub(final Int subtrahend) {
		return new Int(this.value.subtract(subtrahend.value));
	}
	
	/**
	 * Performs the multiplication of two Ints.
	 * 
	 * @param factor
	 *            Another Int object which will be multiplied with.
	 * @return The product of this Int and the other Int.
	 */
	public Int mul(final Int factor) {
		return new Int(this.value.multiply(factor.value));
	}
	
	/**
	 * Performs the integer division of two Ints. The possible remainder of the division is discarded.
	 * 
	 * @param divisor
	 *            Another Int object which will be used as the divisor.
	 * @return The quotient of this Int and the other Int.
	 */
	public Int div(final Int divisor) {
		return new Int(this.value.divide(divisor.value));
	}
	
	/**
	 * Compares one Int object to another and checks if this is less than or equal to another.
	 * 
	 * @param compareTo
	 *            The other Int object.
	 * @return True if this is less than or equal to compareTo.
	 */
	public boolean lessEq(final Int compareTo) {
		return this.value.compareTo(compareTo.value) <= 0;
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof Int) {
			final Int other = (Int) obj;
			return this.value.equals(other.value);
		}
		return false;
	}
	
	/**
	 * Returns the integer value as a BigInteger.
	 * 
	 * @return The integer value as a BigInteger.
	 */
	public BigInteger getVal() {
		return this.value;
	}
	
	@Override
	public int hashCode() {
		return this.value.hashCode();
	}
	
	@Override
	public String toString() {
		return this.value.toString();
	}
	
}
