package de.fhdw.wtf.context.core;

import java.io.IOException;
import java.lang.reflect.Modifier;

import de.fhdw.wtf.context.exception.DatabaseManagerFactoryNotInstantiableException;
import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.context.model.collections.PersistentList;
import de.fhdw.wtf.context.model.collections.PersistentListFactory;
import de.fhdw.wtf.context.model.collections.PersistentMap;
import de.fhdw.wtf.context.model.collections.PersistentMapFactory;
import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.TypeOrAssociationNotFoundException;
import de.fhdw.wtf.persistence.facade.ClassFacade;
import de.fhdw.wtf.persistence.facade.DatabaseManager;
import de.fhdw.wtf.persistence.facade.DatabaseManagerFactory;
import de.fhdw.wtf.persistence.facade.TypeManager;
import de.fhdw.wtf.persistence.utils.PropertiesReader;
import de.fhdw.wtf.persistence.utils.PropertiesReaderFactory;
import de.fhdw.wtf.persistence.utils.PropertiesReaderFile;

/**
 * An application starter configures the successful launch of the complete application. It initializes the Database
 * Manager, the logger and the object factories. A developer has to subclass this class and provide the location of
 * configuration files etc.
 *
 */
public abstract class ApplicationStarter {
	
	/**
	 * The delimiter to separate packages in a path.
	 */
	private static final String PACKAGE_DELIMITER = ".";
	
	/**
	 * The path to search for DatabaseManagers.
	 */
	private static final String SOURCE_PACKAGE_OF_DATABASE_MANAGER_FACTORIES = "de" + PACKAGE_DELIMITER + "fhdw"
			+ PACKAGE_DELIMITER + "wtf" + PACKAGE_DELIMITER + "persistence" + PACKAGE_DELIMITER + "facade";
	
	/**
	 * 
	 */
	public static final Object ORACLE_DATABASE_MANAGER_FACTORY_NAME = "OracleDatabaseManagerFactory";
	
	/**
	 * This method launches the application in a standalone java environment. After calling this method, the Application
	 * Container can be used as central application registry.
	 */
	public void startStandalone() {
		this.start(this.getResourcesPathJava());
	}
	
	/**
	 * This method launches the application for deployment inside an JEE container. After calling this method, the
	 * Application Container can be used as central application registry.
	 */
	public void startServer() {
		this.start(this.getResourcesPathServer());
	}
	
	/**
	 * This method launches the application. After calling this method, the object facade can be used.
	 * 
	 * @param resourcesPath
	 *            Needs a reference to the location of the application configuration files.
	 */
	public void start(final String resourcesPath) {
		PropertiesReader prop;
		try {
			prop = new PropertiesReaderFile();
			prop.initialize(resourcesPath + "/" + this.getApplicationConfigFileName());
			// fileStream = new FileInputStream(new File(resourcesPath + "/" + this.getApplicationConfigFileName()));
		} catch (final Exception e) {
			throw new FrameworkException(e.getMessage());
		}
		try {
			// final Properties applicationProperties = new Properties();
			// applicationProperties.load(fileStream);
			this.initializeLogger();
			ApplicationContainer.getInstance().setAppName(prop.getProperty("application-name"));
			ApplicationContainer.getInstance().setUsedDatabaseManagerFactoryName(prop.getProperty("database"));
			this.initializeDatabase(resourcesPath, ApplicationContainer.getInstance()
					.getUsedDatabaseManagerFactoryName());
			this.initializeRuntimePersistence();
			
			// ClassFacade classFacade =
			// ApplicationContainer.getInstance().getDatabaseManager().getClassFacade();
			// UserType mutableSet =
			// classFacade.createUserType("de>fhdw>wtf>context>model>collections>MutableList",
			// false, false);
			// UserType persistentSet =
			// classFacade.createUserType("de>fhdw>wtf>context>model>collections>PersistentList",
			// false, false);
			// UserType mutableMap =
			// classFacade.createUserType("de>fhdw>wtf>context>model>collections>MutableMap",
			// false, false);
			// UserType persistentMap =
			// classFacade.createUserType("de>fhdw>wtf>context>model>collections>PersistentMap",
			// false, false);
			//
			// classFacade.createSpecializationBetween(mutableSet,
			// persistentSet);
			// classFacade.createSpecializationBetween(mutableMap,
			// persistentMap);
			
			this.initializeFactories();
			this.registerActivities();
		} catch (final PersistenceException e) {
			throw new FrameworkException(e.getMessage());
		} catch (final IOException e) {
			throw new FrameworkException(e.getMessage());
		} finally {
		}
	}
	
	/**
	 * This method represents a hook, to enable the programmer to define the application specific Activities and the
	 * according View Facades.
	 */
	protected abstract void registerActivities();
	
	/**
	 * This method stops the application. The database connection will be closed.
	 */
	public void stop() {
		try {
			ApplicationContainer.getInstance().getDatabaseManager().disconnect();
		} catch (final PersistenceException e) {
			throw new FrameworkException(e.getMessage());
		}
	}
	
	/**
	 * This method initializes the user provides Object factories.
	 * 
	 * @throws ClassFacadeUninitializedException
	 *             If the ClassFacade has not been initialized before, this exception will be thrown.
	 * @throws TypeOrAssociationNotFoundException
	 *             If Type created by a factory is not known inside the database, an exception will be thrown.
	 */
	private void initializeFactories() throws ClassFacadeUninitializedException, TypeOrAssociationNotFoundException {
		final ObjectFactoryProvider factoryProvider = ObjectFactoryProvider.instance();
		ApplicationContainer.getInstance().setFactoryProvider(factoryProvider);
		final TypeManager typeManager =
				ApplicationContainer.getInstance().getDatabaseManager().getClassFacade().getTypeManager();
		try {
			factoryProvider.registerTypeFactory(
					typeManager,
					PersistentList.class.getName(),
					new PersistentListFactory());
		} catch (final TypeOrAssociationNotFoundException e) {
			/* if the application does not use lists, no factory is necessary */
		}
		try {
			factoryProvider.registerTypeFactory(typeManager, PersistentMap.class.getName(), new PersistentMapFactory());
		} catch (final TypeOrAssociationNotFoundException e) {
			/* if the application does not use maps, no factory is necessary */
		}
		this.registerTypeFactories(factoryProvider, typeManager);
	}
	
	/**
	 * This method must be implemented by a subclass to register the type factories to instantiate user defined types.
	 * 
	 * @param factoryProvider
	 *            The holder of the different factories, on it an the register method will be called.
	 * @param typeManager
	 *            The Type Manager can be used to provide the id of a type for which a factory is created.
	 * @throws TypeOrAssociationNotFoundException
	 *             If Type created by a factory is not known inside the database, an exception will be thrown.
	 */
	protected abstract void registerTypeFactories(ObjectFactoryProvider factoryProvider, TypeManager typeManager)
			throws TypeOrAssociationNotFoundException;
	
	/**
	 * This method initializes the Persistence for Runtime.
	 * 
	 * @throws PersistenceException
	 *             An exception will be thrown if there are any exception when contacting the database.
	 */
	private void initializeRuntimePersistence() throws PersistenceException {
		final ClassFacade classFacade = ApplicationContainer.getInstance().getDatabaseManager().getClassFacade();
		classFacade.initializeForRuntime();
	}
	
	/**
	 * This method initializes the logger component.
	 */
	private void initializeLogger() {
		ApplicationContainer.getInstance().setLogger(Logger.getInstance());
	}
	
	/**
	 * This method will initialize the Database Manager and its connection with the Database.
	 * 
	 * @param resourcesPath
	 *            Path where the resources for the DBMS connection are stored.
	 * @param databaseManagerFactoryClassName
	 *            A String which is the value of the parameter 'database' in the central configuration file. It should
	 *            be identical to the name of a concrete type which is a descendant of {@link DatabaseManagerFactory}.
	 * @throws IOException
	 *             An exception will be thrown if the database configuration file could not be read,
	 * @throws PersistenceException
	 *             if some database error occurs.
	 */
	private void initializeDatabase(final String resourcesPath, final String databaseManagerFactoryClassName)
			throws IOException, PersistenceException {
		Class<?> newClass;
		try {
			newClass =
					Class.forName(ApplicationStarter.SOURCE_PACKAGE_OF_DATABASE_MANAGER_FACTORIES + PACKAGE_DELIMITER
							+ databaseManagerFactoryClassName);
		} catch (final ClassNotFoundException e) {
			throw new DatabaseManagerFactoryNotInstantiableException(e.getMessage());
		}
		if (Modifier.isAbstract(newClass.getModifiers()) || !DatabaseManagerFactory.class.isAssignableFrom(newClass)) {
			throw new DatabaseManagerFactoryNotInstantiableException(
					DatabaseManagerFactoryNotInstantiableException.CLASS_IS_A_CONCRETE_DATABASE_MANAGER_FACTORY_REASON);
		}
		DatabaseManager database;
		try {
			database = ((DatabaseManagerFactory) newClass.newInstance()).getInstance();
		} catch (final InstantiationException | IllegalAccessException e) {
			throw new DatabaseManagerFactoryNotInstantiableException(e.getMessage());
		}
		
		// database.setConnectionConstantsFromFile(resourcesPath + "/" + this.getDatabaseConfigFileName());
		database.setConnectionConstantsFromFile(PropertiesReaderFactory.getInstance().getPropertiesReaderOracle());
		database.connect();
		ApplicationContainer.getInstance().setDatabaseManager(database);
		
	}
	
	/**
	 * This Path specifies the directory where all the config files reside. It has to be specified by the developer of
	 * an application. By convention the path is: $(ProjectRoot)/src/main/resources/config
	 * 
	 * @return A String specifying the resources path.
	 */
	protected abstract String getResourcesPathJava();
	
	/**
	 * This Path specified the directory where all the config files reside when packaged as a Web Archive for deployment
	 * on a server. It has to be specified by the developer of an application. By convention path is: WEB-INF/config
	 * 
	 * @return A string spcifyong the resources path.
	 */
	protected abstract String getResourcesPathServer();
	
	/**
	 * The file name of the central application config file. In this file the database provider and other essential
	 * informations are stored. By convention it is named: application.properties
	 * 
	 * @return A String representing the File Name of the central application config file.
	 */
	protected abstract String getApplicationConfigFileName();
	
	/**
	 * The file name of the config file containing all needed information for the chosen persistence provider.
	 * 
	 * @return A String representing the file Name of the Persistence config file.
	 */
	protected abstract String getDatabaseConfigFileName();
	
	/**
	 * The package hierarchy to the root path of the model classes. It represents the technical prefix all model classes
	 * get, when placed in this package. By convention it is: generated.model
	 * 
	 * @return A String representing the model prefix.
	 */
	protected abstract String getModelPrefix();
	
}
