package de.fhdw.wtf.context.model.collections;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.Iterator;

import de.fhdw.wtf.context.core.ObjectFactoryProvider;
import de.fhdw.wtf.context.core.TransactionManager;
import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.context.model.AnyType;
import de.fhdw.wtf.context.model.Anything;
import de.fhdw.wtf.context.model.Int;
import de.fhdw.wtf.context.model.Str;
import de.fhdw.wtf.persistence.meta.MapLink;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * This class represents Maps. It contains 3 valued associations with one value K as key and another value V as value.
 * Values can be accessed via the key.
 * 
 * @param <K>
 *            Type of key elements.
 * @param <V>
 *            Type of value elements.
 */
public class PersistentMapWithKeyValueLinks<K extends Anything, V extends Anything> implements Map<K, V> {
	
	/**
	 * Maps keys to MapLinks and the values behind the links.
	 */
	private final HashMap<K, Tuple<MapLink, V>> mapCache;
	
	/**
	 * The name of the association to which this collection belongs.
	 */
	private final String associationName;
	
	/**
	 * The owner object of this association.
	 */
	private final UserObject owner;
	
	/**
	 * Creates a PersistentMapWithKeyValueLinks.
	 * 
	 * @param owner
	 *            The owner of the association.
	 * @param associationName
	 *            The name of the association.
	 */
	public PersistentMapWithKeyValueLinks(final UserObject owner, final String associationName) {
		this.owner = owner;
		this.associationName = associationName;
		this.mapCache = new HashMap<>();
	}
	
	@Override
	public void put(final K key, final V value) {
		if (key == null || value == null) {
			throw new NullPointerException();
		}
		// set value in DB
		// TODO visitor für anything
		MapLink returnedLink;
		if (key instanceof Str) {
			final String castedKeyString = ((Str) key).toString();
			if (value instanceof Str) {
				final Str castedValue = (Str) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.toString(),
								castedKeyString);
			} else if (value instanceof Int) {
				final Int castedValue = (Int) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.getVal(),
								castedKeyString);
			} else if (value instanceof AnyType) {
				final AnyType castedValue = (AnyType) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.getObject(),
								castedKeyString);
			} else {
				throw new FrameworkException("Type to insert is not known");
			}
		} else if (key instanceof Int) {
			final BigInteger castedKeyObject = ((Int) key).getVal();
			if (value instanceof Str) {
				final Str castedValue = (Str) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.toString(),
								castedKeyObject);
			} else if (value instanceof Int) {
				final Int castedValue = (Int) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.getVal(),
								castedKeyObject);
			} else if (value instanceof AnyType) {
				final AnyType castedValue = (AnyType) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.getObject(),
								castedKeyObject);
			} else {
				throw new FrameworkException("Type to insert is not known");
			}
		} else if (key instanceof AnyType) {
			final UserObject castedKeyObject = ((AnyType) key).getObject();
			if (value instanceof Str) {
				final Str castedValue = (Str) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.toString(),
								castedKeyObject);
			} else if (value instanceof Int) {
				final Int castedValue = (Int) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.getVal(),
								castedKeyObject);
			} else if (value instanceof AnyType) {
				final AnyType castedValue = (AnyType) value;
				returnedLink =
						TransactionManager.getContext().put(
								this.owner,
								this.associationName,
								castedValue.getObject(),
								castedKeyObject);
			} else {
				throw new FrameworkException("Type to insert is not known");
			}
		} else {
			throw new FrameworkException("Type to insert is not known");
		}
		// set value in internal map
		this.mapCache.put(key, new Tuple<>(returnedLink, value));
	}
	
	@Override
	public V get(final K key) {
		if (key == null) {
			throw new NullPointerException();
		}
		// if key is in internal map => get from internal map
		// otherwise get from db
		if (this.mapCache.containsKey(key)) {
			return this.getFromCache(key);
		} else {
			java.util.Collection<Tuple<MapLink, Object>> databaseResult = null;
			if (key instanceof Str) {
				databaseResult =
						TransactionManager.getContext().get(this.owner, this.associationName, ((Str) key).toString());
			} else if (key instanceof Int) {
				databaseResult =
						TransactionManager.getContext().get(this.owner, this.associationName, ((Int) key).getVal());
			} else if (key instanceof AnyType) {
				databaseResult =
						TransactionManager.getContext().get(
								this.owner,
								this.associationName,
								((AnyType) key).getObject());
			} else {
				throw new FrameworkException("Type to insert is not known");
			}
			final Iterator<Tuple<MapLink, Object>> iterator = databaseResult.iterator();
			// No results
			if (!iterator.hasNext()) {
				this.mapCache.put(key, null);
				return null;
			}
			// at least one result
			final Tuple<MapLink, Object> firstResult = iterator.next();
			final Object firstResultTarget = firstResult.getSecond();
			@SuppressWarnings("unchecked")
			final V result = (V) ObjectFactoryProvider.instance().createObject(firstResultTarget);
			this.mapCache.put(key, new Tuple<>(firstResult.getFirst(), result));
			return this.getFromCache(key);
		}
	}
	
	/**
	 * Tries to look up a key in the MapLinks map.
	 * 
	 * @param key
	 *            The key.
	 * @return The value or null if not found.
	 */
	private V getFromCache(final K key) {
		final Tuple<MapLink, V> localTupel = this.mapCache.get(key);
		if (localTupel == null) {
			return null;
		}
		return localTupel.getSecond();
	}
	
	@Override
	public V remove(final K key) {
		if (key == null) {
			throw new NullPointerException();
		}
		final Tuple<MapLink, V> toRemoveTuple = this.mapCache.get(key);
		// remove in DB
		TransactionManager.getContext().unset(toRemoveTuple.getFirst());
		// remove local, but keep in mind that the key was received from DB
		this.mapCache.put(key, null);
		return toRemoveTuple.getSecond();
	}
	
	// @Override
	// public void clear() {
	// //clear local map
	// //clear DB map
	// //database operation needed (not possible to do client side, because all
	// keys would be retrieved)
	// }
	
}
