package de.fhdw.wtf.context.core;

/**
 * A class to represent the Transaction Manager, which is used to hold the thread local Context.
 */
public final class TransactionManager {
	
	/**
	 * Private default constructor to disallow instances of this class.
	 */
	private TransactionManager() {
		// nothing to do
	}
	
	/**
	 * The thread local storage which holds the Context.
	 */
	private static ThreadLocal<Context> contextHolder = new ThreadLocal<Context>() {
		@Override
		protected Context initialValue() {
			return new NoContext();
		}
	};
	
	/**
	 * Provides the actual Context.
	 * 
	 * @return The currently saved Context of the Thread.
	 */
	public static Context getContext() {
		return contextHolder.get();
	}
	
	/**
	 * This methods sets a provided Context in the scope of this thread.
	 * 
	 * @param context
	 *            A context which will be the new actual Context in this thread.
	 */
	public static void setContext(final Context context) {
		contextHolder.set(context);
	}
	
	/**
	 * This method commits the current context, after the commit the new Context will be a NoContext.
	 */
	public static void commit() {
		contextHolder.get().commit();
		contextHolder.set(new NoContext());
	}
	
	/**
	 * This method rolls the current context back, after the commit the new Context will be a NoContext.
	 */
	public static void rollback() {
		contextHolder.get().rollback();
		contextHolder.set(new NoContext());
	}
	
}
