package de.fhdw.wtf.context.core;

import java.text.SimpleDateFormat;
import java.util.Date;

import de.fhdw.wtf.persistence.exception.PersistenceException;

/**
 * A class to represent the central Logging instance, which can be used for debugging, analysing and monitoring
 * purposes.
 */
public final class Logger {
	
	/**
	 * The format for the time.
	 */
	private final SimpleDateFormat format = new SimpleDateFormat("dd.MM.yyyy HH:MM:ss:SSSS");
	
	/**
	 * The static field needed for singleton pattern.
	 */
	private static Logger instance = null;
	
	/**
	 * Static getInstance Method (Singleton Pattern).
	 * 
	 * @return Provides the Instance of the Logger.
	 */
	public static synchronized Logger getInstance() {
		if (instance == null) {
			instance = new Logger();
		}
		return new Logger();
	}
	
	/**
	 * Private default Logger constructor.
	 */
	private Logger() {
	}
	
	/**
	 * Provides the default prefix for a Log message. The current time with the severity of the message appended.
	 * 
	 * @param severity
	 *            The severity of the Message
	 * @return Provides a String representing the prefix of a Log entry.
	 */
	private String getLoggerPrefix(final String severity) {
		return "[" + this.format.format(new Date()) + ";" + severity + "] :";
	}
	
	/**
	 * Logs an Persistence Exception to Sys.err.
	 * 
	 * @param e
	 *            The occured Persistence Exception.
	 */
	public void log(final PersistenceException e) {
		System.err.println(this.getLoggerPrefix("Exception") + e.getMessage());
	}
	
	/**
	 * Logs an error to Sys.err.
	 * 
	 * @param message
	 *            The message describing the error.
	 */
	public void logError(final String message) {
		System.err.println(this.getLoggerPrefix("Error") + message);
	}
	
	/**
	 * Logs an information message to sys.out.
	 * 
	 * @param message
	 *            The message containing information.
	 */
	public void logInfo(final String message) {
		System.out.println(this.getLoggerPrefix("Info") + message);
	}
	
}
