package de.fhdw.wtf.context.model.collections;

import java.util.Iterator;
import java.util.Vector;

import de.fhdw.wtf.context.core.TransactionManager;
import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.context.model.AnyType;
import de.fhdw.wtf.context.model.Anything;
import de.fhdw.wtf.context.model.Int;
import de.fhdw.wtf.context.model.Str;
import de.fhdw.wtf.context.model.collections.functors.Predicate;
import de.fhdw.wtf.context.model.collections.functors.Procedure;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * A class to represent a mutable list of objects with persistence integration. It also represents an
 * UnidirectionalAssociation therefore all changes on this collection will directly affect the associated owner object.
 * 
 * @param <T>
 *            Any Type T, which is subtype of Anything.
 */
public class PersistentListWithLinks<T extends Anything> implements MutableCollection<T> {
	
	/**
	 * A collection of Links which is used to operate on this association.
	 */
	private final java.util.Collection<Tuple<UnidirectionalLink, Object>> unidirectionalLinks;
	
	/**
	 * The name of the association to which this collection belongs.
	 */
	private final String associationName;
	
	/**
	 * The owner object of this association.
	 */
	private final UserObject owner;
	
	/**
	 * Creates a new PersistentListWithLinks object.
	 * 
	 * @param unidirectionalLinks
	 *            The unidirectionalLinks, which are the elements to the association.
	 * @param associationName
	 *            The name of the association.
	 * @param owner
	 *            The owner of the association.
	 */
	public PersistentListWithLinks(final java.util.Collection<Tuple<UnidirectionalLink, Object>> unidirectionalLinks,
			final String associationName,
			final UserObject owner) {
		super();
		this.unidirectionalLinks = unidirectionalLinks;
		this.associationName = associationName;
		this.owner = owner;
	}
	
	@Override
	public Collection<T> union(final Collection<? extends T> otherCollection) {
		if (otherCollection instanceof ImmutableCollection) {
			this.addImmutableCollection((ImmutableCollection<? extends T>) otherCollection);
		} else if (otherCollection instanceof MutableCollection) {
			this.addMutableCollection((MutableCollection<? extends T>) otherCollection);
		} else {
			throw new FrameworkException("Collection is not supported");
		}
		return this;
	}
	
	/**
	 * Adds the elements of a MutableCollection to this list.
	 * 
	 * @param otherCollection
	 *            The MutableCollection.
	 */
	private void addMutableCollection(final MutableCollection<? extends T> otherCollection) {
		final Iterator<? extends T> iterator = otherCollection.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			this.insert(current);
		}
	}
	
	/**
	 * Adds the elements of an ImmutableCollection to this list.
	 * 
	 * @param otherCollection
	 *            The ImmutableCollection.
	 */
	private void addImmutableCollection(final ImmutableCollection<? extends T> otherCollection) {
		if (!otherCollection.isEmpty()) {
			this.insert(otherCollection.front());
			this.addImmutableCollection(otherCollection.tail());
		}
	}
	
	@Override
	public boolean contains(final T element) {
		final Iterator<T> iterator = this.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			if (element.equals(current)) {
				return true;
			}
		}
		return false;
	}
	
	@Override
	public void insert(final T element) {
		if (element instanceof AnyType) {
			final AnyType toAdd = (AnyType) element;
			final UnidirectionalLink nw =
					TransactionManager.getContext().set(this.owner, this.associationName, toAdd.getObject());
			this.unidirectionalLinks.add(new Tuple<>(nw, nw.getTarget()));
		} else if (element instanceof Int) {
			final Int toAdd = (Int) element;
			final UnidirectionalLink nw =
					TransactionManager.getContext().set(this.owner, this.associationName, toAdd.getVal());
			this.unidirectionalLinks.add(new Tuple<>(nw, nw.getTarget()));
		} else if (element instanceof Str) {
			final Str toAdd = (Str) element;
			final UnidirectionalLink nw =
					TransactionManager.getContext().set(this.owner, this.associationName, toAdd.toString());
			this.unidirectionalLinks.add(new Tuple<>(nw, nw.getTarget()));
		} else {
			throw new FrameworkException("Type to insert is not known");
		}
		
	}
	
	@Override
	public void apply(final Procedure<T> procedure) {
		final Iterator<T> iterator = this.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			procedure.op(current);
		}
	}
	
	@Override
	public void remove(final Predicate<T> predicate) {
		final Iterator<T> iterator = this.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			if (predicate.p(current)) {
				iterator.remove();
			}
		}
	}
	
	@Override
	public Iterator<T> iterator() {
		return new PersistenceIteratorWithLink<>(this.unidirectionalLinks);
	}
	
	@Override
	public boolean isEmpty() {
		return this.unidirectionalLinks.isEmpty();
	}
	
	@Override
	public ImmutableCollection<T> copy() {
		ImmutableCollection<T> result = new ImmutableList<>();
		final Iterator<T> iterator = this.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			result = result.add(current);
		}
		return result;
	}
	
	@Override
	public String toString() {
		final Vector<T> result = new Vector<>();
		final Iterator<T> i = this.iterator();
		while (i.hasNext()) {
			final T current = i.next();
			result.add(current);
		}
		return result.toString();
	}
	
	@Override
	public boolean equals(final java.lang.Object obj) {
		if (obj instanceof MutableCollection<?>) {
			try {
				@SuppressWarnings("unchecked")
				final MutableCollection<T> other = (MutableCollection<T>) obj;
				final Iterator<T> thisIterator = this.iterator();
				while (thisIterator.hasNext()) {
					final T current = thisIterator.next();
					if (!other.contains(current)) {
						return false;
					}
				}
				final Iterator<T> otherIterator = other.iterator();
				while (otherIterator.hasNext()) {
					final T current = otherIterator.next();
					if (!this.contains(current)) {
						return false;
					}
				}
				return true;
			} catch (final ClassCastException e) {
				return false;
			}
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return this.copy().hashCode();
	}
}
